/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas;

import org.apache.axis2.context.ConfigurationContext;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.wso2.utils.FileManipulator;
import org.wso2.utils.ServerConfiguration;
import org.wso2.utils.ServerConfigurationException;
import org.wso2.utils.ServerException;
import org.wso2.wsas.util.Monitor;
import org.wso2.wsas.util.ServerController;
import org.wso2.wsas.util.WsasUtils;
import org.wso2.wsas.util.XmlConfiguration;

import java.io.File;
import java.io.IOException;
import java.util.Map;

/**
 * The main class of the WSO2 WSAS standalone runtime.
 */
public class Main {
    private static final Log log = LogFactory.getLog(Main.class);

    // Server lifecycle commands
    public static final String COMMAND_START = "START";
    public static final String COMMAND_RESTART = "RESTART";
    public static final String COMMAND_STOP = "STOP";
    public static final String COMMAND_RUN = "RUN";
    public static final String COMMAND_STATUS = "STATUS";
    public static final String COMMAND_VERSION = "VERSION";
    public static final String COMMAND_DEBUG = "DEBUG";

    public static Thread shutdownHookThread;

    private WebServer webServer;
    private static ServerConfiguration serverConfig;

    public static XmlConfiguration xmlConfiguration;

    public Main() {
        if (System.getProperty(ServerConstants.WSO2WSAS_HOME) == null) {
            System.setProperty(ServerConstants.WSO2WSAS_HOME, ".");
            System.setProperty(ServerConstants.AXIS2_HOME, ".");
        } else {
            System.setProperty(ServerConstants.AXIS2_HOME,
                               System.getProperty(ServerConstants.WSO2WSAS_HOME));
        }
        String serverXMLLocation = WsasUtils.getWsasServerXml();
        try {
            serverConfig = ServerConfiguration.getInstance();
            serverConfig.init(serverXMLLocation);
        } catch (ServerConfigurationException e) {
            throw new RuntimeException(e);
        }
        if (System.getProperty(ServerConstants.DERBY_HOME) == null) {
            System.setProperty(ServerConstants.DERBY_HOME,
                               serverConfig.getFirstProperty("Database.Home"));
        }
    }


    public void startServer() {
        try {
            webServer = new TomcatServer();
            webServer.start();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public void stop() throws IOException, ServerException {
        ServerManager.getInstance().stop();
        stopListeners();
    }

    public void status() throws IOException {
        ServerManager.getInstance().status();
    }

    public void shutdown() throws ServerException {
        stopListeners();
        cleanupSystem();
    }

    public static void main(String[] args) {
        System.setProperty(ServerConstants.WSO2WSAS_START_TIME,
                           String.valueOf(System.currentTimeMillis()));
        String cmd = null;

        // Set the System properties
        for (int i = 0; i < args.length; i++) {
            String arg = args[i];
            if (arg.startsWith("-D")) {
                int indexOfEq = arg.indexOf("=");
                String property;
                String value;
                if (indexOfEq != -1) {
                    property = arg.substring(2, indexOfEq);
                    value = arg.substring(indexOfEq + 1);
                } else {
                    property = arg.substring(2);
                    value = "true";
                }
                System.setProperty(property, value);
            } else {
                if (cmd == null) {
                    cmd = arg;
                } else if (!cmd.toUpperCase().endsWith(COMMAND_DEBUG)) {
                    printUsages();
                    return;
                }
            }
        }

        if (cmd == null) {
            cmd = COMMAND_RUN;
        }

        final Main main = new Main();
        String serverName = serverConfig.getFirstProperty("Name");

        if (cmd.toUpperCase().endsWith(COMMAND_START) ||
            cmd.toUpperCase().endsWith(COMMAND_RUN) ||
            cmd.toUpperCase().endsWith(COMMAND_DEBUG)) {
            if (cmd.toUpperCase().endsWith(COMMAND_DEBUG)) {
                log.info("Starting " + serverName + " " +
                         serverConfig.getFirstProperty("Version") +
                         " (Powered by Apache Axis2) in debug mode");
            } else {
                log.info("Starting " + serverName + " " +
                         serverConfig.getFirstProperty("Version") + " (Powered by Apache Axis2)");
            }
            log.info("Using Java Home        : " + System.getProperty("java.home"));
            log.info("Using Java Version     : " + System.getProperty("java.version"));
            log.info("Using " + serverName + " Home   : " + System.getProperty("wso2wsas.home"));

            main.startServer();
        } else if (cmd.toUpperCase().endsWith(COMMAND_RESTART)) {
            try {
                ServerController serverController = new ServerController();
                serverController.setPort(WsasUtils.getCommandListenerPort());
                serverController.restartServer();
            } catch (Exception e) {
                String msg = "Cannot restart " + serverName;
                log.error(msg, e);
            }
        } else if (cmd.toUpperCase().endsWith(COMMAND_STOP)) {
            try {
                ServerController serverController = new ServerController();
                serverController.setPort(WsasUtils.getCommandListenerPort());
                serverController.shutDownServer();
            } catch (Exception e) {
                String msg = "Cannot stop " + serverName;
                log.error(msg, e);
            }
        } else if (cmd.toUpperCase().endsWith(COMMAND_VERSION)) {
            System.out.println(serverName + " version: " + serverConfig.getFirstProperty("Version"));
        } else if (cmd.toUpperCase().endsWith(COMMAND_STATUS)) {
            try {
                main.status();
            } catch (IOException e) {
                String msg = "Cannot get status of " + serverName;
                log.error(msg, e);
            }
        } else {
            printUsages();
        }
    }

    private static void printUsages() {
        String serverName = serverConfig.getFirstProperty("Name");
        System.out.println("Usage: " + System.getProperty("server.script") + " [command]");
        System.out.println("command:");
        System.out.println("\t--start\t\tStart " + serverName + " as a background process");
        System.out.println("\t--run\t\tRun " + serverName);
        System.out.println("\t--restart\tRestart " + serverName);
        System.out.println("\t--stop\t\tStop " + serverName);
        System.out.println("\t--debug <port> \tStart " + serverName + " in remote debugging mode." +
                           "\n\t\t\tport: The remote debugging port.");
        System.out.println("\t--version\tWhat version of " + serverName + " are you running?");
        System.out.println();
    }

    public ConfigurationContext getConfigContext() {
        return ServerManager.getInstance().configContext;
    }

    public void setMonitor(Monitor m) {
        ServerManager.getInstance().monitor = m;
    }

    public void stopListeners() throws ServerException {
        if (webServer != null) {
            try {
                webServer.stop();
            } catch (Exception axisFault) {
                throw new ServerException(axisFault);
            }
        }
        webServer = null;
    }

    private void cleanupSystem() {
        log.info("Cleaning up system...");
        new FileManipulator().deleteDir(new File(serverConfig.getFirstProperty("WorkDirectory")));
        ConfigurationContext configContext = ServerManager.getInstance().configContext;
        if (configContext != null) {
            Object property = configContext.getProperty(ServerConstants.FILE_RESOURCE_MAP);
            if (property != null) {
                ((Map) property).clear();
            }
        }
    }
}
