/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas;

import org.apache.axiom.om.OMElement;
import org.apache.axis2.AxisFault;
import org.apache.axis2.context.ConfigurationContext;
import org.apache.axis2.description.AxisModule;
import org.apache.axis2.description.Parameter;
import org.apache.axis2.description.TransportInDescription;
import org.apache.axis2.engine.AxisConfiguration;
import org.apache.axis2.i18n.Messages;
import org.apache.axis2.util.Loader;
import org.apache.commons.collections.BidiMap;
import org.apache.commons.collections.bidimap.TreeBidiMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.log4j.Logger;
import org.wso2.adminui.UIProcessingException;
import org.wso2.adminui.UIProcessor;
import org.wso2.tracer.TracerConstants;
import org.wso2.tracer.module.TracePersister;
import org.wso2.utils.ManagementFactory;
import org.wso2.utils.NetworkUtils;
import org.wso2.utils.ServerConfiguration;
import org.wso2.utils.ServerException;
import org.wso2.utils.security.CryptoException;
import org.wso2.utils.security.CryptoUtil;
import org.wso2.wsas.persistence.PersistenceManager;
import org.wso2.wsas.persistence.dataobject.ModuleDO;
import org.wso2.wsas.persistence.dataobject.SecurityScenarioDO;
import org.wso2.wsas.persistence.dataobject.ServiceUserDO;
import org.wso2.wsas.persistence.dataobject.ServiceUserRoleDO;
import org.wso2.wsas.persistence.dataobject.TransportDO;
import org.wso2.wsas.persistence.exception.SecurityScenarioAlreadyExistsException;
import org.wso2.wsas.persistence.exception.ServiceUserAlreadyExistsException;
import org.wso2.wsas.persistence.exception.TransportAlreadyExistsException;
import org.wso2.wsas.persistence.exception.UserRoleAlreadyExistsException;
import org.wso2.wsas.security.WSASJMXAuthenticator;
import org.wso2.wsas.util.ClusteringUtil;
import org.wso2.wsas.util.HouseKeepingTask;
import org.wso2.wsas.util.XmlConfiguration;

import javax.management.MBeanServer;
import javax.management.remote.JMXConnectorServer;
import javax.management.remote.JMXConnectorServerFactory;
import javax.management.remote.JMXServiceURL;
import javax.xml.namespace.QName;
import java.io.File;
import java.rmi.registry.LocateRegistry;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;
import java.util.Timer;

/*
 * The default initializer which initilizes WSAS
 */
public class DefaultServerInitializer implements ServerInitializer {
    private static final Log log = LogFactory.getLog(DefaultServerInitializer.class);

    private ServerConfiguration serverConfig;
    private PersistenceManager pm;
    private String wso2wsasHome;
    private static Logger rootLogger = Logger.getRootLogger();
    private boolean isHouseKeepingTaskCreated;
    private Timer houseKeepingTimer;
    private ServerManager serverManager;
    private static boolean isJMXServiceStarted;

    public DefaultServerInitializer() {
        this.serverConfig = ServerConfiguration.getInstance();
        this.serverManager = ServerManager.getInstance();
        this.pm = new PersistenceManager();
    }


    public void init(ConfigurationContext configurationContext) throws AxisFault, ServerException {

        wso2wsasHome = System.getProperty(ServerConstants.WSO2WSAS_HOME);

        initHouseKeeping();

        // enabled clustering
        enableClustering(configurationContext);

        // Create the dynamic pages and save it in the ConfigContext
        generatePages(configurationContext);

        // Enabling http binding generation
        Parameter enableHttp = new Parameter("enableHTTP", "true");
        AxisConfiguration axisConfig = configurationContext.getAxisConfiguration();
        axisConfig.addParameter(enableHttp);


        configurationContext.setProperty(ServerConstants.Logging.MEMORY_APPENDER,
                                         rootLogger.getAppender(ServerConstants.
                                                 Logging.WSO2WSAS_MEMORY_APPENDER));

        createDefaultServiceUserRoles();

        createDefaultAdminAccount();

        handleGlobalModuleEngagements(axisConfig);

        initServiceGroupContextTimeout(configurationContext);

        registerHouseKeepingTask(configurationContext);

        persistSecurityScenarios(configurationContext);

        initKeystoresDir();

        enableSoapTracing(axisConfig);

        persistTransports(axisConfig);

        log.info("");
        log.info("Repository           : " + serverManager.axis2RepoLocation);
        startJMXService();
    }

    private void startJMXService() throws ServerException {
        String jmxPort = ServerConfiguration.getInstance().getFirstProperty("Ports.JMX");
        if (jmxPort != null) {
            if (isJMXServiceStarted) {
                return;
            }
            int jmxPortInt = Integer.parseInt(jmxPort);
            MBeanServer mbs = ManagementFactory.getMBeanServer();
            try {
                LocateRegistry.createRegistry(jmxPortInt);

                // Create an RMI connector and start it
                String jmxURL = "service:jmx:rmi:///jndi/rmi://" +
                                NetworkUtils.getLocalHostname() + ":" + jmxPortInt + "/server";
                JMXServiceURL url = new JMXServiceURL(jmxURL);

                // Security credentials are included in the env Map
                HashMap env = new HashMap();
                env.put(JMXConnectorServer.AUTHENTICATOR, new WSASJMXAuthenticator());
                JMXConnectorServer cs =
                        JMXConnectorServerFactory.newJMXConnectorServer(url, env, mbs);
                cs.start();
                log.info("JMX Service URL      : " + jmxURL);
                isJMXServiceStarted = true;
            } catch (Exception e) {
                String msg = "Could not initialize MBean server";
                log.error(msg, e);
                throw new ServerException(msg, e);
            }
        }
    }

    private void enableSoapTracing(AxisConfiguration axisConfig) throws AxisFault {
        // Enable SOAP Tracing if necessary
        AxisModule tracerModule = axisConfig.getModule(TracerConstants.WSO2_TRACER);
        if (tracerModule != null) {
            Parameter tracePersisterParam =
                    axisConfig.getParameter(TracerConstants.TRACE_PERSISTER_IMPL);
            TracePersister tracePersister = null;
            if (tracePersisterParam != null) {
                Object tracePersisterImplObj = tracePersisterParam.getValue();
                if (tracePersisterImplObj instanceof TracePersister) {
                    tracePersister = (TracePersister) tracePersisterImplObj;
                } else if (tracePersisterImplObj instanceof String) {
                    //This will need in TestSuite
                    try {
                        tracePersister =
                                (TracePersister) Loader.loadClass(((String) tracePersisterImplObj).trim())
                                        .newInstance();
                    } catch (Exception e) {
                        String message = "Cannot instatiate TracePersister ";
                        log.error(message, e);
                        throw new RuntimeException(message, e);
                    }
                }
            } else {
                throw new AxisFault(TracerConstants.TRACE_PERSISTER_IMPL +
                                    " parameter not defined in axis2.xml");
            }
            if (tracePersister != null && tracePersister.isTracingEnabled()) {
                if (!axisConfig.isEngaged(tracerModule)) {
                    axisConfig.engageModule(tracerModule);
                }
            } else {
                if (axisConfig.isEngaged(tracerModule)) {
                    axisConfig.disengageModule(tracerModule);
                }
            }
        }
    }

    private void enableClustering(ConfigurationContext configurationContext) throws AxisFault {
        ClusteringUtil.enableClustering(configurationContext);
    }

    private void generatePages(ConfigurationContext configurationContext) {
        Map files = new Hashtable();
        try {
            UIProcessor.createPages(serverManager.adminResourceBase, "ui-extensions-config.xml",
                                    files);
            configurationContext.setProperty(ServerConstants.GENERATED_PAGES, files);
        } catch (UIProcessingException e) {
            String msg = "Static page generation failed";
            log.warn(msg + ": " + e);
            if (log.isDebugEnabled()) {
                log.debug(msg, e);
            }
        }
    }

    private void createDefaultServiceUserRoles() throws ServerException {
        OMElement document = serverConfig.getDocumentElement();

        for (Iterator surIte =
                document.getChildrenWithName(
                        new QName(ServerConstants.WSO2WSAS_XML_NAMESPACE, "ServiceUserRoles"));
             surIte.hasNext();) {

            OMElement serviceUserRoles = (OMElement) surIte.next();

            for (Iterator roleIte =
                    serviceUserRoles.getChildrenWithName(
                            new QName(ServerConstants.WSO2WSAS_XML_NAMESPACE, "Role"));
                 roleIte.hasNext();) {
                OMElement role = (OMElement) roleIte.next();
                ServiceUserRoleDO serviceUserRole = null;
                for (Iterator nameIter =
                        role.getChildrenWithName(
                                new QName(ServerConstants.WSO2WSAS_XML_NAMESPACE, "Name"));
                     nameIter.hasNext();) {
                    OMElement name = (OMElement) nameIter.next();
                    serviceUserRole = pm.getServiceUserRole(name.getText());
                    if (serviceUserRole == null) {
                        serviceUserRole = new ServiceUserRoleDO();
                        serviceUserRole.setRole(name.getText());
                    }
                }
                for (Iterator nameIter =
                        role.getChildrenWithName(
                                new QName(ServerConstants.WSO2WSAS_XML_NAMESPACE, "Description"));
                     nameIter.hasNext();) {
                    OMElement description = (OMElement) nameIter.next();
                    if (serviceUserRole != null) {
                        serviceUserRole.setDescription(description.getText());
                    } else {
                        throw new ServerException("Description without a Role name is invalid");
                    }
                }
                try {
                    pm.addOrUpdateUserRole(serviceUserRole);
                } catch (UserRoleAlreadyExistsException ignored) {
                    // this exception cannot occur since we've validated it above
                }
            }
        }
    }

    private void createDefaultAdminAccount() throws ServerException {
        String adminUsername = "admin";
        ServiceUserDO admin = pm.getUser(adminUsername);
        ServerConfiguration config = ServerConfiguration.getInstance();
        CryptoUtil cryptoUtil =
                new CryptoUtil(new File(
                        config.getFirstProperty("Security.KeyStore.Location")).getAbsolutePath(),
                               config.getFirstProperty("Security.KeyStore.Password"),
                               config.getFirstProperty("Security.KeyStore.KeyAlias"),
                               config.getFirstProperty("Security.KeyStore.KeyPassword"),
                               config.getFirstProperty("Security.KeyStore.Type"));
        if (admin == null) {

            // There is no default admin user. WSAS is starting up for the first time
            // Check whether the default Admin user's password has been changed
            String adminPassword = getDefaultAdminPassword();
            String adminDescription = "Default Adminstrator";
            try {
                admin = new ServiceUserDO();
                admin.setUsername(adminUsername);
                admin.setPassword(cryptoUtil.encryptAndBase64Encode(adminPassword.getBytes()));
                admin.setDescription(adminDescription);
                ServiceUserRoleDO role = pm.getUserRole(ServerConstants.ADMIN_ROLE);
                pm.addUser(admin);
                pm.addRole(admin.getUsername(), role);
            } catch (CryptoException e) {
                String msg = "Cannot encrypt default admin password";
                log.error(msg, e);
                throw new ServerException(msg, e);
            } catch (ServiceUserAlreadyExistsException ignored) {
                // this exception cannot occur since we've validated it above
            }
        } else {
            try {
                if (new String(cryptoUtil.base64DecodeAndDecrypt(admin.getPassword())).
                        equals("admin")) {
                    String password = getDefaultAdminPassword();
                    admin.setPassword(cryptoUtil.encryptAndBase64Encode(password.getBytes()));
                    pm.updateUser(admin);
                }
            } catch (Exception e) {
                throw new ServerException(e);
            }
        }
    }


    private void persistTransports(AxisConfiguration axisConfig) throws AxisFault {

        PersistenceManager pm = new PersistenceManager();

        // Check whether the user has removed transports from Axis2.xml
        TransportDO[] transports = pm.getTransports();
        for (int i = 0; i < transports.length; i++) {
            TransportDO transport = transports[i];
            if (axisConfig.
                    getTransportIn(transport.getTransportProtocol()) == null) {
                if (!transport.getServices().isEmpty()) {
                    String msg = "Cannot remove the " + transport.getTransportProtocol() + " transport.\n" +
                                 "\t\t\t\tOne or more services are bound to this transport. \n" +
                                 "\t\t\t\tAdd this transport back to the axis2.xml file and restart the server.";
                    throw new AxisFault(msg);
                } else {
                    pm.deleteEntity(transport);
                }
            }
        }

        // Persist the transport protocols
        for (Iterator transportsIter =
                axisConfig.getTransportsIn().values().iterator();
             transportsIter.hasNext();) {
            TransportInDescription tiDesc = (TransportInDescription) transportsIter.next();
            String protocolName = tiDesc.getName();
            if (pm.getTransport(protocolName) == null) { // Transport does not already exist?
                TransportDO transportDO = new TransportDO();
                transportDO.setTransportProtocol(protocolName);
                transportDO.setIsStarted(true);
                try {
                    pm.addTransport(transportDO);
                } catch (TransportAlreadyExistsException ignored) {
                    // ignored
                }
            }
        }
    }

    private void handleGlobalModuleEngagements(AxisConfiguration axisConfig)
            throws ServerException {
        try {
            ModuleDO[] moduleDOs = pm.getAllModules();
            for (int i = 0; i < moduleDOs.length; i++) {
                ModuleDO moduleDO = moduleDOs[i];
                if (moduleDO.getIsGloballyEngaged()) {
                    String moduleId = moduleDO.getModuleIdentifierDO().getName();
                    if (!axisConfig.isEngaged(moduleId)) {
                        axisConfig.engageModule(moduleId);
                    }

                    // We should not engage the throttle module to the wso2wsas-administration service group
                    if (moduleId.equals("wso2throttle")) {
                        axisConfig.getServiceGroup(ServerConstants.ADMIN_SERVICE_GROUP).
                                disengageModule(axisConfig.getModule(moduleId));
                    }
                }
            }
        } catch (Exception e) {
            String msg = Messages.getMessage("CannotConfigureAxis2");
            log.error(msg);
            throw new ServerException(msg, e);
        }
    }

    private void initServiceGroupContextTimeout(ConfigurationContext configurationContext) {
        String serviceGroupContextIdleTimeout =
                serverConfig.getFirstProperty("Axis2Config.ServiceGroupContextIdleTime");

        if (serviceGroupContextIdleTimeout != null) {
            configurationContext.
                    setProperty(org.apache.axis2.Constants.Configuration.
                            CONFIG_CONTEXT_TIMOUT_INTERVAL,
                                new Integer(serviceGroupContextIdleTimeout));
        }
    }

    private void registerHouseKeepingTask(ConfigurationContext configurationContext) {
        if (!isHouseKeepingTaskCreated) {
            if (Boolean.valueOf(serverConfig.
                    getFirstProperty("HouseKeeping.AutoStart")).booleanValue()) {
                houseKeepingTimer = new Timer();
                long houseKeepingInterval =
                        Long.parseLong(serverConfig.
                                getFirstProperty("HouseKeeping.Interval")) * 60 * 1000;
                Object property =
                        configurationContext.getProperty(ServerConstants.FILE_RESOURCE_MAP);
                if (property == null) {
                    property = new TreeBidiMap();
                    configurationContext.setProperty(ServerConstants.FILE_RESOURCE_MAP, property);
                }
                houseKeepingTimer
                        .scheduleAtFixedRate(new HouseKeepingTask(serverManager.serverWorkDir,
                                                                  (BidiMap) property),
                                             houseKeepingInterval,
                                             houseKeepingInterval);
                isHouseKeepingTaskCreated = true;
            }
        }
    }

    private void persistSecurityScenarios(ConfigurationContext configurationContext)
            throws ServerException {
        String scenarioConfigXml = wso2wsasHome + File.separator + "conf" +
                                   File.separator + "rampart" +
                                   File.separator + "scenario-config.xml";
        if (!new File(scenarioConfigXml).exists()) {
            log.info("Security scenario configuration file does not exist");
            return;
        }
        XmlConfiguration xmlConfiguration =
                new XmlConfiguration(scenarioConfigXml,
                                     "http://www.wso2.org/products/wsas/security");
        OMElement[] secenarioEle = xmlConfiguration.getElements("//ns:Scenario");
        for (int i = 0; i < secenarioEle.length; i++) {
            SecurityScenarioDO scenario = new SecurityScenarioDO();
            OMElement omElement = secenarioEle[i];
            String scenarioId =
                    omElement.getAttribute(ServerConstants.Security.ID_QN).getAttributeValue();
            if (pm.getSecurityScenario(scenarioId) != null) {
                continue;
            }
            scenario.setScenarioId(scenarioId);
            scenario.setSummary(omElement.
                    getFirstChildWithName(ServerConstants.Security.SUMMARY_QN).getText());
            scenario.setDescription(omElement
                    .getFirstChildWithName(ServerConstants.Security.DESCRIPTION_QN).getText());
            scenario.setCategory(omElement
                    .getFirstChildWithName(ServerConstants.Security.CATEGORY_QN).getText());

            for (Iterator modules = omElement
                    .getFirstChildWithName(ServerConstants.Security.MODULES_QN).getChildElements();
                 modules.hasNext();) {
                String module = ((OMElement) modules.next()).getText();
                AxisModule axisModule =
                        configurationContext.getAxisConfiguration().getModule(module);

                if (axisModule != null) {
                    ModuleDO moduleDO = pm.getModule(axisModule.getName(), axisModule.getVersion());
                    scenario.addModule(moduleDO);
                }
            }
            try {
                pm.addSecurityScenario(scenario);
            } catch (SecurityScenarioAlreadyExistsException ignored) {
                // will not be able to add an already existing Security Scenario
            }
        }
    }


    private void initKeystoresDir() {
        ServerConfiguration serverConfig = ServerConfiguration.getInstance();
        String ksDirName = serverConfig.getFirstProperty("Security.KeyStoresDir");
        serverConfig.setConfigurationProperty("Security.KeyStoresDir",
                                              new File(ksDirName).getAbsolutePath());
    }


    private void initHouseKeeping() {
        if (houseKeepingTimer != null) {
            houseKeepingTimer.cancel();
            houseKeepingTimer = null;
            isHouseKeepingTaskCreated = false;
        }
    }

    /**
     * There is no default admin user. WSAS is starting up for the first time
     * Check whether the default Admin user's password has been changed
     *
     * @return Default admin password
     */
    private String getDefaultAdminPassword() {
        return "admin";

        //TODO: We need to finalize on how we are to handle capturing of the default admin password
        /*String adminPassword = System.getProperty(ServerConstants.WSO2WSAS_ADMIN_PASSWORD);
       if (adminPassword != null && !adminPassword.equals("admin")) {
           return adminPassword;
       }
       System.out.println(" Please enter the password of Administrator 'admin'");
       String passwordRepeat = null;
       do {
           while (adminPassword == null || adminPassword.trim().length() == 0) {
               try {
                   adminPassword = InputReader.readPassword(" New password: ");
                   if (adminPassword.equals("admin")) {
                       adminPassword = null;
                       System.err.println(" This password is not allowed." +
                                          " Please enter another password.");
                   }
               } catch (IOException e) {
                   System.err.println(" Unable to read password : " + e);
               }
           }
           while (passwordRepeat == null || passwordRepeat.trim().length() == 0) {
               try {
                   passwordRepeat = InputReader.readPassword(" Re-enter new password: ");
               } catch (IOException e) {
                   System.err.println(" Unable to read re-entered password : " + e);
               }
           }
           if (!adminPassword.equals(passwordRepeat)) {
               System.err.println(" Password and re-entered password do not match");
               adminPassword = null;
               passwordRepeat = null;
               continue;
           }
           System.out.println(" Password for Admin user changed.");
           break;
       } while (true);
       return adminPassword;*/
    }

    public void stopHouseKeeping() {
        initHouseKeeping();
    }


}
