/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.clustering.configuration.commands;

import org.apache.axis2.context.ConfigurationContext;
import org.apache.axis2.deployment.DeploymentEngine;
import org.apache.axis2.description.AxisService;
import org.apache.axis2.description.AxisServiceGroup;
import org.apache.axis2.engine.AxisConfiguration;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.wso2.utils.ServerConfiguration;
import org.wso2.wsas.util.WsasUtils;

import javax.activation.DataHandler;
import java.io.File;
import java.io.FileOutputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Random;

/**
 * Loads a new Service Group
 */
public class LoadServiceGroupsCommand extends AbstractConfigurationCommand {
    private static Log log = LogFactory.getLog(LoadServiceGroupsCommand.class);

    private List axisServiceGroups = new ArrayList();
    private String[] serviceGroupNames;
    private static final Random RANDOM = new Random();

    public synchronized void process(ConfigurationContext configCtx) throws Exception {
        log.info("Going to load service groups...");
        axisServiceGroups.clear();
        AxisConfiguration axisConfig = configCtx.getAxisConfiguration();

        ServerConfiguration serverConfig = ServerConfiguration.getInstance();
        String axis2Repo =
                serverConfig.getFirstProperty(ServerConfiguration.AXIS2_CONFIG_REPO_LOCATION);
        for (int i = 0; i < serviceGroupNames.length; i++) {
            String sgName = serviceGroupNames[i];
            if (!sgName.endsWith(".aar")) {
                sgName += ".aar";
            }
            log.info("Going to load service group - " + sgName + " ...");
            AxisServiceGroup serviceGroup = axisConfig.getServiceGroup(sgName);

            if (serviceGroup == null) {  // New service group
                File serviceArchive;
                if (WsasUtils.isURL(axis2Repo)) {
                    DataHandler dh = new DataHandler(new URL(axis2Repo + "services/" + sgName));
                    String tempDir =
                            serverConfig.getFirstProperty("WorkDirectory") + File.separator +
                            (System.currentTimeMillis() + RANDOM.nextDouble());
                    new File(tempDir).mkdirs();
                    serviceArchive = new File(tempDir + File.separator + sgName);
                    FileOutputStream out = new FileOutputStream(serviceArchive);
                    dh.writeTo(out);
                    out.close();
                } else {
                    serviceArchive = new File(axis2Repo + File.separator + "services" +
                                              File.separator + sgName);
                }
                AxisServiceGroup asGroup =
                        DeploymentEngine.loadServiceGroup(serviceArchive, configCtx);
                axisServiceGroups.add(asGroup);
            } else { // Existing service group
                //TODO Do we unload the service group & reload it, ignore this request or throw an error
                String msg = "Service group " + sgName + " already exists!";
                log.error(msg);
                throw new Exception(msg);
            }
        }
    }

    public synchronized void prepare(ConfigurationContext configContext) {
        for (Iterator iterator = this.axisServiceGroups.iterator();
             iterator.hasNext();) {
            AxisServiceGroup axisSG = (AxisServiceGroup) iterator.next();
            addBlockAllRequestsParameter(axisSG);
        }
    }

    public synchronized void commit(ConfigurationContext configContext) throws Exception {
        log.info("Committing " + this + "...");
        try {
            for (Iterator iter = axisServiceGroups.iterator(); iter.hasNext();) {
                AxisServiceGroup serviceGroup = (AxisServiceGroup) iter.next();
                configContext.getAxisConfiguration().addServiceGroup(serviceGroup);
                removeBlockAllRequestsParameter(serviceGroup);
            }
        } finally {
            axisServiceGroups.clear();
        }
        log.info("Committed " + this);
    }

    public synchronized void rollback(ConfigurationContext configContext) throws Exception {
        log.info("Rolling back " + this + "...");
        AxisConfiguration axisConfig = configContext.getAxisConfiguration();
        try {
            for (int i = 0; i < axisServiceGroups.size(); i++) {
                AxisServiceGroup serviceGroup = (AxisServiceGroup) axisServiceGroups.get(i);
                for (Iterator serviceIter = serviceGroup.getServices(); serviceIter.hasNext();) {
                    AxisService service = (AxisService) serviceIter.next();
                    if (axisConfig.getService(service.getName()) != null) {
                        axisConfig.removeService(service.getName());
                    }
                }
                String serviceGroupName = serviceGroup.getServiceGroupName();
                if (axisConfig.getServiceGroup(serviceGroupName) != null) {
                    axisConfig.removeServiceGroup(serviceGroupName);
                }
            }
        } finally {
            axisServiceGroups.clear();
        }
        log.info("Rolled back " + this + "...");
    }

    public String toString() {
        return "LOAD_SERVICE_GROUPS_EVENT";
    }

    public synchronized void setServiceGroupNames(String[] serviceGroupNames) {
        this.serviceGroupNames = serviceGroupNames;
    }

    public int getCommandType() {
        return LOAD_SERVICE_GROUPS;
    }
}
