/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.clustering.configuration.commands;

import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.apache.axis2.context.ConfigurationContext;
import org.apache.axis2.description.AxisService;
import org.apache.axis2.description.PolicyInclude;
import org.apache.neethi.Policy;
import org.apache.neethi.util.PolicyComparator;
import org.wso2.wsas.util.PolicyUtil;

import java.io.ByteArrayInputStream;
import java.util.Iterator;

/**
 *
 */
public class ApplyServicePolicyCommand extends AbstractConfigurationCommand {

    private ServicePolicy servicePolicy;

    private String policy;
    private String serviceName;

    public void process(ConfigurationContext confiCtx) throws Exception {
        log.info("Applying policy to service ...");
        this.servicePolicy = null;

        // Build the Policy
        ByteArrayInputStream bais = new ByteArrayInputStream(policy.getBytes());
        OMElement policyElement;
        AxisService axisService =
                confiCtx.getAxisConfiguration().getService(serviceName);
        if (axisService == null) {
            String msg = "Service " + serviceName + " does not exist";
            log.info(msg);
            throw new Exception(msg);
        }
        policyElement = new StAXOMBuilder(bais).getDocumentElement();
        policyElement.build();
        Policy policy = PolicyUtil.getPolicyFromOMElement(policyElement);
        this.servicePolicy = new ServicePolicy(axisService, policy);
    }

    public void prepare(ConfigurationContext configContext) {
        addBlockAllRequestsParameter(this.servicePolicy.getAxisService());
    }

    public void commit(ConfigurationContext configContext) throws Exception {
        AxisService axisService = this.servicePolicy.getAxisService();
        Policy newPolicy = this.servicePolicy.getPolicy();

        PolicyInclude policyInclude = axisService.getPolicyInclude();

        // Handle existing policy updates
        boolean isPolicyFound = false;
        for (Iterator iter = policyInclude.getPolicyElements().iterator();
             iter.hasNext();) {
            Object obj = iter.next();
            if (obj instanceof Policy) {
                Policy policy = (Policy) obj;
                if (PolicyComparator.compare(policy, newPolicy)) {
                    newPolicy.setId(policy.getId());
                    policyInclude.updatePolicy(newPolicy);
                    isPolicyFound = true;
                    break;
                }
            }
        }

        // Handle new policy additions
        if (!isPolicyFound) {
            policyInclude.addPolicyElement(PolicyInclude.AXIS_SERVICE_POLICY, newPolicy);
        }
        removeBlockAllRequestsParameter(axisService);
    }

    public void rollback(ConfigurationContext configContext) throws Exception {
    }

    public String toString() {
        return "APPLY_SERVICE_POLICY_EVENT";
    }

    public int getCommandType() {
        return APPLY_SERVICE_POLICY;
    }

    public void setPolicy(String policy) {
        this.policy = policy;
    }

    public void setServiceName(String serviceName) {
        this.serviceName = serviceName;
    }

    private static class ServicePolicy {
        private AxisService axisService;
        private Policy policy;

        public ServicePolicy(AxisService axisService, Policy policy) {
            this.axisService = axisService;
            this.policy = policy;
        }

        public AxisService getAxisService() {
            return axisService;
        }

        public Policy getPolicy() {
            return policy;
        }
    }
}

