/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.clustering.configuration;

import org.apache.axis2.clustering.configuration.ConfigurationManagerListener;
import org.apache.axis2.clustering.configuration.ConfigurationClusteringCommand;
import org.apache.axis2.clustering.ClusteringConstants;
import org.apache.axis2.context.ConfigurationContext;
import org.apache.axis2.description.AxisService;
import org.apache.axis2.description.AxisServiceGroup;
import org.apache.axis2.description.Parameter;
import org.apache.axis2.engine.AxisConfiguration;
import org.apache.axis2.AxisFault;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.wso2.wsas.clustering.NodeManager;

import java.util.Iterator;
import java.util.List;
import java.util.Vector;

/**
 *
 */
public class WSASConfigurationManagerListener implements ConfigurationManagerListener {

    private static Log log = LogFactory.getLog(WSASConfigurationManagerListener.class);

    private ConfigurationContext configurationContext;

    private List pendingCommits = new Vector();

    /**
     * A configuration change request has been received
     */
    private boolean isProcessing;

    /**
     * System is preparing to commit to a configuration change
     */
    private boolean isPreparing;

    /**
     * System has received a commit configuration request, and is in the process of switching to the
     * new configuration
     */
    private boolean isCompleteReceived;

    /**
     * The max waiting period(ms) after receiving a prepare. The system waits for
     * DEFAULT_COMMIT_TIMEOUT ms before reverting to the old configuration and continuing to service
     * Web service requests
     */
    private static final int DEFAULT_COMMIT_TIMEOUT = 50000;

    /**
     * Keeps track of whether a commit timer is currently running
     */
    private boolean isCommitTimerStarted;

    /**
     * Thread which rolls back the configuration if a commit message is not received within a
     * specified time
     */
    private Thread commitTimer;

    public void serviceGroupsLoaded(ConfigurationClusteringCommand cmd) {
        processCommand(cmd);
    }

    public void serviceGroupsUnloaded(ConfigurationClusteringCommand cmd) {
        processCommand(cmd);
    }

    public void policyApplied(ConfigurationClusteringCommand cmd) {
        processCommand(cmd);
    }

    /**
     * This callback method notifies this object that other nodes are reloading a configuration
     *
     * @param cmd  The ConfigurationClusteringCommand
     */
    public void configurationReloaded(ConfigurationClusteringCommand cmd) {
        processCommand(cmd);
    }

    private void processCommand(ConfigurationClusteringCommand cmd) {

        // Set isProcessing to true so that, even if a prepare or commit is received
        // before this method is completed, the system will wait, until the reload is complete
        isProcessing = true;
        try {
            pendingCommits.add(cmd);
            startCommitTimer(System.currentTimeMillis());
            cmd.process(configurationContext);
        } catch (Throwable e) {
            notifyFailure(e);
        }

        // We've finished processing this request, so reset the isProcessing flag to false
        isProcessing = false;
    }

    /**
     * Prepare the system to change configuration The system will block all service requests, in the
     * case of a LOAD_CONFIGURATION_EVENT, or block all requests to a particular service, in the
     * case of other events.
     */
    public void prepareCalled() {
        isPreparing = true;
        Thread prepareWaiter = new Thread("PrepareWaiter") {
            public void run() {
                while (isProcessing) {
                    try {
                        Thread.sleep(200);
                    } catch (InterruptedException ignored) {
                        ignored.printStackTrace();
                    }
                }
                if (pendingCommits.isEmpty()) {
                    log.warn("Prepare command received but no pending commits found.");
                    notifyFailureToNodeManager();
                    return;
                }
                log.info("Preparing to commit...");
                for (Iterator iter = pendingCommits.iterator(); iter.hasNext();) {
                    ((ConfigurationClusteringCommand) iter.next()).prepare(configurationContext);
                }
                isPreparing = false;
            }
        };
        prepareWaiter.start();
    }

    private void removeCommitInProgressParam() {
        try {
            AxisService axisService =
                    configurationContext.getAxisConfiguration().
                            getService(ClusteringConstants.NODE_MANAGER_SERVICE);
            Parameter parameter = axisService.
                    getParameter(NodeManager.COMMIT_IN_PROGRESS);
            if (parameter != null) {
                axisService.removeParameter(parameter);
            }
        } catch (AxisFault e) {
            log.error("Error occurred while removing commit in progress parameter", e);
        }
    }

    private void startCommitTimer(final long startedTime) {
        if (isCommitTimerStarted) {
            return;
        }
        // If a commit is not received within x seconds of a prepare being received,
        // we need to discard what was done earlier and start serving requests
        commitTimer = new Thread("CommitTimer") {
            public void run() {
                isCommitTimerStarted = true;
                AxisConfiguration axisConfig = configurationContext.getAxisConfiguration();
                Parameter parameter =
                        axisConfig.getClusterManager().getConfigurationManager().
                                getParameter("CommitTimeout");
                long timeOut = DEFAULT_COMMIT_TIMEOUT;
                if (parameter != null) {
                    timeOut = Long.parseLong((String) parameter.getValue());
                }
                while (!isCompleteReceived) {
                    if (startedTime != 0 &&
                        System.currentTimeMillis() - startedTime > timeOut) {
                        pendingCommits.clear();
                        configurationContext.
                                removePropertyNonReplicable(ClusteringConstants.BLOCK_ALL_REQUESTS);
                        log.info("Commit/Rollback message not received within " + timeOut +
                                 " ms. Rolling back all changes & resuming operations.");
                        rollback();
                        break;
                    }
                    try {
                        Thread.sleep(2000);
                    } catch (InterruptedException ignored) {
                        break;
                    }
                }
                isCommitTimerStarted = false;
                isCompleteReceived = false;
            }
        };
        commitTimer.start();
    }

    /**
     */
    public void rollbackCalled() {
        notifyFailureToNodeManager();
        isCompleteReceived = true;
        commitTimer.interrupt();
        log.info("Rolling back configuration changes...");
        try {
            rollback();
        } catch (Exception e) {
            log.error("Exception occurred while rolling back configuration changes", e);
        }
        log.info("Configuration changes rolled back successfully.");
    }

    public void commitCalled() {
        commitTimer.interrupt();
        isCompleteReceived = true;
        Thread commitThread = new Thread() {
            public void run() {
                if (pendingCommits.isEmpty()) {
                    log.warn("Commit command received but no pending commits found.");
                    notifyFailureToNodeManager();
                    removeCommitInProgressParam();
                    return;
                }
                while (isPreparing) {
                    try {
                        Thread.sleep(200);
                    } catch (InterruptedException ignored) {
                        ignored.printStackTrace();
                    }
                }

                // Wait for a few more seconds, since a handleException message could be received
                try {
                    Thread.sleep(2000);
                } catch (InterruptedException ignored) {
                    ignored.printStackTrace();
                }
                log.info("Committing configuration changes...");
                for (Iterator iter = pendingCommits.iterator(); iter.hasNext();) {
                    ConfigurationClusteringCommand cmd =
                            (ConfigurationClusteringCommand) iter.next();
                    try {
                        cmd.commit(configurationContext);
                        log.info("Commit successful");
                    } catch (Exception e) {
                        log.info("Commit failed");
                        notifyFailure(new Exception("Could not commit " + cmd));
                    }
                }
                configurationContext.
                        removePropertyNonReplicable(ClusteringConstants.BLOCK_ALL_REQUESTS);
                pendingCommits.clear();
                removeCommitInProgressParam();
            }
        };
        commitThread.start();
    }

    public void handleException(Throwable throwable) {
        log.info("Handling exception...");
        notifyFailureToNodeManager();
        commitTimer.interrupt();
        isCompleteReceived = true;
        rollback();
    }

    public void setConfigurationContext(ConfigurationContext configurationContext) {
        this.configurationContext = configurationContext;
    }

    private void rollback() {
        for (Iterator iter = pendingCommits.iterator(); iter.hasNext();) {
            ConfigurationClusteringCommand cmd = (ConfigurationClusteringCommand) iter.next();
            try {
                cmd.rollback(configurationContext);
            } catch (Exception e) {
                log.error("Error occurred while rolling back command " + cmd, e);
            }
        }
        pendingCommits.clear();
        removeCommitInProgressParam();
        isCompleteReceived = false;
        isPreparing = false;
        isProcessing = false;
        configurationContext.removePropertyNonReplicable(ClusteringConstants.BLOCK_ALL_REQUESTS);
        for (Iterator iter = configurationContext.getAxisConfiguration().getServiceGroups();
             iter.hasNext();) {
            AxisServiceGroup serviceGroup = (AxisServiceGroup) iter.next();
            Parameter parameter = serviceGroup.getParameter(ClusteringConstants.BLOCK_ALL_REQUESTS);
            if (parameter != null) {
                try {
                    serviceGroup.removeParameter(parameter);
                } catch (AxisFault e) {
                    log.error("Could not remove the " + ClusteringConstants.BLOCK_ALL_REQUESTS +
                              " parameter from the " + serviceGroup.getServiceGroupName() +
                              " service group.");
                }
            }
        }
    }

    private void notifyFailure(Throwable e) {

        try {
            notifyFailureToNodeManager();

            // Notify other nodes
            configurationContext.getAxisConfiguration().
                    getClusterManager().getConfigurationManager().exceptionOccurred(e);
        } catch (Exception e2) {
            log.error("Error occurred while notifying failure to cluster or local NodeManager", e2);
        }
    }

    private void notifyFailureToNodeManager() {
        // Notify NodeManager service
        try {
            AxisService axisService =
                    configurationContext.getAxisConfiguration().
                            getService(ClusteringConstants.NODE_MANAGER_SERVICE);
            if (axisService != null) {
                axisService.addParameter(new Parameter(NodeManager.OPERATION_FAILED, "true"));
            }
        } catch (Exception e2) {
            log.error("Error occurred while notifying failure to local node manager", e2);
        }
    }
}
