/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.clustering;

import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.apache.axis2.AxisFault;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

/**
 *
 */
public class NodeManagerClient {
    private WSO2WSASNodeManagerStub stub;

    public NodeManagerClient(String nodeManagerEPR) throws AxisFault {
        stub = new WSO2WSASNodeManagerStub(nodeManagerEPR);
        stub._getServiceClient().getOptions().setManageSession(true);
        setSecurityProperties();
    }

    private boolean reloadConfiguration() throws Exception {
        System.out.println("Going to reload configuration...");
        return stub.reloadConfiguration().get_return();
    }

    private boolean unloadServiceGroups(String[] serviceGroupNames) throws Exception {
        for (int i = 0; i < serviceGroupNames.length; i++) {
            System.out.println("Unloading service group: " + serviceGroupNames[i]);
        }
        WSO2WSASNodeManagerStub.UnloadServiceGroups req =
                new WSO2WSASNodeManagerStub.UnloadServiceGroups();
        req.setServiceGroupNames(serviceGroupNames);
        WSO2WSASNodeManagerStub.UnloadServiceGroupsResponse resp = stub.unloadServiceGroups(req);
        return resp.get_return();
    }

    private boolean loadServiceGroups(String[] serviceGroupNames) throws Exception {
        for (int i = 0; i < serviceGroupNames.length; i++) {
            System.out.println("Loading service group: " + serviceGroupNames[i]);
        }
        WSO2WSASNodeManagerStub.LoadServiceGroups req =
                new WSO2WSASNodeManagerStub.LoadServiceGroups();
        req.setServiceGroupNames(serviceGroupNames);
        WSO2WSASNodeManagerStub.LoadServiceGroupsResponse resp = stub.loadServiceGroups(req);
        return resp.get_return();
    }

    private boolean applyServicePolicy(String serviceName, String policyFile) throws Exception {
        System.out.println("Applying policy to service: " + serviceName);
        WSO2WSASNodeManagerStub.ApplyPolicy req =
                new WSO2WSASNodeManagerStub.ApplyPolicy();
        req.setServiceName(serviceName);
        StAXOMBuilder builder = new StAXOMBuilder(new FileInputStream(policyFile));
        req.setPolicy(builder.getDocumentElement());
        WSO2WSASNodeManagerStub.ApplyPolicyResponse resp = stub.applyPolicy(req);
        return resp.get_return();
    }

    private boolean commit() throws Exception {
        System.out.println("Committing...");
        if (stub.prepare().get_return()) {
            return stub.commit().get_return();
        }
        return false;
    }

    private boolean login(String username, String password) throws Exception {
        System.out.println("Logging in...");
        WSO2WSASNodeManagerStub.Login req = new WSO2WSASNodeManagerStub.Login();
        req.setUsername(username);
        req.setPassword(password);
        return stub.login(req).get_return();
    }

    /**
     * @param args
     */
    public static void main(String[] args) {
        try {
            String username = getParam("--username", args);
            String password = getParam("--password", args);
            String epr = getParam("--epr", args);
            NodeManagerClient client = new NodeManagerClient(epr);

            String operation = getParam("--operation", args);

            // Login
            if (!client.login(username, password)) {
                System.err.println("Login failed. Please verify the admin username & password");
            }

            boolean isOperationSuccessful = false;
            if (operation.equals("reloadconfig")) {
                isOperationSuccessful = client.reloadConfiguration();
            } else if (operation.equals("loadsgs")) {
                String[] serviceGroups = getServiceGroups(getParam("--service-groups", args));
                isOperationSuccessful = client.loadServiceGroups(serviceGroups);
            } else if (operation.equals("unloadsgs")) {
                String[] serviceGroups = getServiceGroups(getParam("--service-groups", args));
                isOperationSuccessful = client.unloadServiceGroups(serviceGroups);
            } else if (operation.equals("applypolicy")) {
                String serviceName = getParam("--service", args);
                String policyFile = getParam("--policy-file", args);
                isOperationSuccessful = client.applyServicePolicy(serviceName, policyFile);
            } else if (operation.equals("commit")) {
                isOperationSuccessful = client.commit();
            } else {
                printUsage();
            }

            // Set the correct exit status
            if (isOperationSuccessful) {
                System.exit(0);
            } else {
                System.exit(1);
            }
        } catch (Exception e) {
            e.printStackTrace();
            System.exit(1);
        }
    }

    private static String[] getServiceGroups(String serviceGroups) {
        StringTokenizer tokenizer = new StringTokenizer(serviceGroups, ",");
        List list = new ArrayList();
        while (tokenizer.hasMoreTokens()) {
            list.add(tokenizer.nextToken().trim());
        }
        return (String[]) list.toArray(new String[list.size()]);
    }

    private static void printUsage() {
        System.out.println("usage: admin --username <username> --password <password> --epr <epr> " +
                           "--operation <reloadconfig|loadsgs|unloadsgs|applypolicy|commit> " +
                           "[options]");
        System.out.println("options: --service-groups <ServiceGroupName1,ServiceGroupName2,...,ServiceGroupNameN>");
    }

    /**
     * This will check the given parameter in the array and will return, if
     * available
     *
     * @param param
     * @param args
     * @return String param
     */
    private static String getParam(String param, String[] args) {
        if ((param == null) || "".equals(param)) {
            return null;
        }

        for (int i = 0; i < args.length; i = i + 2) {
            String arg = args[i];
            if (param.equalsIgnoreCase(arg) && (args.length >= (i + 1))) {
                if (args.length == i + 1) {
                    System.err.println("Invalid value specified for option " + arg);
                    printUsage();
                    System.exit(1);
                }
                return args[i + 1].trim();
            }
        }
        printUsage();
        System.exit(1);
        return null;
    }

    private void setSecurityProperties() {
        String ksPassword = System.getProperty("javax.net.ssl.keyStorePassword");
        if (ksPassword == null) {
            System.setProperty("javax.net.ssl.keyStorePassword", "wso2wsas");
        }

        String ksType = System.getProperty("javax.net.ssl.keyStoreType");
        if (ksType == null) {
            System.setProperty("javax.net.ssl.keyStoreType", "JKS");
        }

        String trustStore = System.getProperty("javax.net.ssl.trustStore");

        if (trustStore == null || trustStore.trim().length() == 0) {
            System.setProperty("javax.net.ssl.trustStore",
                               ".." + File.separator + "conf" + File.separator +
                               "client-truststore.jks");

        }

        String tsPassword = System.getProperty("javax.net.ssl.trustStorePassword");
        if (tsPassword == null) {
            System.setProperty("javax.net.ssl.trustStorePassword", "wso2wsas");
        }
    }
}
