/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.clustering;

import org.apache.axiom.om.OMElement;
import org.apache.axis2.AxisFault;
import org.apache.axis2.clustering.ClusterManager;
import org.apache.axis2.clustering.configuration.ConfigurationManager;
import org.apache.axis2.context.ConfigurationContext;
import org.apache.axis2.context.MessageContext;
import org.apache.axis2.description.AxisService;
import org.apache.axis2.description.Parameter;
import org.apache.axis2.engine.AxisConfiguration;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.wso2.wsas.admin.service.GlobalAdmin;

/**
 *
 */
public class NodeManager {
    private Log log = LogFactory.getLog(NodeManager.class);
    private static final String CONFIG_MANAGER_NULL =
            "ConfigurationManager is null. Please properly set the 'cluster' element in the axis2.xml file";
    private static final String CLUSTER_MANAGER_NULL =
            "ClusterManager is null. Please properly set the 'cluster' element in the axis2.xml file";
    public static final String OPERATION_FAILED = "nodemanager.operation.failed";
    public static final String COMMIT_IN_PROGRESS = "nodemanager.commit.in.progress";
    private static final int DEFAULT_NOTIFICATION_WAIT_TIME = 2000;

    public boolean unloadServiceGroups(String[] serviceGroupNames) throws AxisFault {
        init();
        ConfigurationManager configMan = getConfigurationManager();
        if (configMan == null) {
            throw new AxisFault(CONFIG_MANAGER_NULL);
        }
        configMan.unloadServiceGroups(serviceGroupNames);
        return isOperationSuccessful();
    }

    public boolean loadServiceGroups(String[] serviceGroupNames) throws AxisFault {
        init();
        ConfigurationManager configMan = getConfigurationManager();
        if (configMan == null) {
            throw new AxisFault(CONFIG_MANAGER_NULL);
        }
        configMan.loadServiceGroups(serviceGroupNames);
        return isOperationSuccessful();
    }

    public boolean applyPolicy(String serviceName, OMElement policy) throws AxisFault {
        init();
        ConfigurationManager configMan = getConfigurationManager();
        if (configMan == null) {
            throw new AxisFault(CONFIG_MANAGER_NULL);
        }
        configMan.applyPolicy(serviceName, policy.getFirstElement().toString());
        return isOperationSuccessful();
    }

    public boolean reloadConfiguration() throws AxisFault {
        init();
        ConfigurationManager configMan = getConfigurationManager();
        if (configMan == null) {
            throw new AxisFault(CONFIG_MANAGER_NULL);
        }
        configMan.reloadConfiguration();
        return isOperationSuccessful();
    }

    public boolean prepare() throws AxisFault {
        return isPrepareSuccessful(getConfigurationManager());
    }

    public boolean commit() throws AxisFault {
        init();
        ConfigurationManager configMan = getConfigurationManager();
        if (configMan == null) {
            throw new AxisFault(CONFIG_MANAGER_NULL);
        }
        return isCommitSuccessful(configMan);
    }

    public boolean login(String username, String password) throws AxisFault {
        return new GlobalAdmin().login(username, password);
    }

    private ConfigurationManager getConfigurationManager() {
        ConfigurationContext rootContext =
                MessageContext.getCurrentMessageContext().getRootContext();

        if (rootContext != null) {
            ClusterManager clusterManager =
                    rootContext.getAxisConfiguration().getClusterManager();
            if (clusterManager != null) {
                ConfigurationManager configMan = clusterManager.getConfigurationManager();
                if (configMan != null) {
                    return configMan;
                }
            } else {
                log.warn(CLUSTER_MANAGER_NULL);
            }
        }
        return null;
    }

    private boolean isCommitSuccessful(ConfigurationManager configMan) throws AxisFault {
        MessageContext msgCtx = MessageContext.getCurrentMessageContext();
        AxisService axisService = msgCtx.getAxisService();
        axisService.addParameter(new Parameter(COMMIT_IN_PROGRESS, "true"));
        Parameter opFailed = axisService.getParameter(OPERATION_FAILED);
        boolean result;
        if (opFailed == null) {
            configMan.commit();
            while (axisService.getParameter(COMMIT_IN_PROGRESS) != null) {
                try {
                    Thread.sleep(1000);
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
            opFailed = axisService.getParameter(OPERATION_FAILED);
            result = (opFailed == null);
        } else {
            result = false;
        }
        if (opFailed != null) {
            axisService.removeParameter(opFailed);
        }
        return result;
    }

    private void init() throws AxisFault {
        MessageContext msgCtx = MessageContext.getCurrentMessageContext();
        AxisService axisService = msgCtx.getAxisService();
        Parameter opFailed = axisService.getParameter(OPERATION_FAILED);
        if (opFailed != null) {
            axisService.removeParameter(opFailed);
        }
    }

    private boolean isOperationSuccessful() throws AxisFault {
        delay(); // Wait for sometime, so that the failures on other nodes can be detected
        MessageContext msgCtx = MessageContext.getCurrentMessageContext();
        AxisService axisService = msgCtx.getAxisService();
        Parameter opFailed = axisService.getParameter(OPERATION_FAILED);
        boolean result;
        if (opFailed == null) {
            delay();
            opFailed = axisService.getParameter(OPERATION_FAILED);
            result = (opFailed == null);
        } else {
            result = false;
        }
        if (opFailed != null) {
            axisService.removeParameter(opFailed);
        }
        return result;
    }

    /**
     * @param configMan
     * @return true if the operation succeeded & prepare succeeded
     * @throws AxisFault
     */
    private boolean isPrepareSuccessful(ConfigurationManager configMan) throws AxisFault {
        delay(); // Wait for sometime, so that the failures on other nodes can be detected
        MessageContext msgCtx = MessageContext.getCurrentMessageContext();
        AxisService axisService = msgCtx.getAxisService();
        Parameter opFailed = axisService.getParameter(OPERATION_FAILED);
        boolean result;
        if (opFailed == null) {
            configMan.prepare(); // Wait for sometime, so that the failures on other nodes can be detected
            delay();
            opFailed = axisService.getParameter(OPERATION_FAILED);
            result = (opFailed == null);
        } else {
            result = false;
        }
        if (opFailed != null) {
            axisService.removeParameter(opFailed);
        }
        return result;
    }

    private void delay() {
        try {
            AxisConfiguration axisConfig =
                    MessageContext.getCurrentMessageContext().getRootContext().getAxisConfiguration();
            Parameter parameter =
                    axisConfig.getClusterManager().getConfigurationManager().
                            getParameter("NotificationWaitTime");
            long delay = DEFAULT_NOTIFICATION_WAIT_TIME;
            if (parameter != null) {
                delay = Long.parseLong((String) parameter.getValue());
            }
            Thread.sleep(delay);
        } catch (InterruptedException e) {
            log.error("Thread was interrupted", e);
        }
    }
}
