/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.admin.service.util;

import org.apache.axis2.AxisFault;
import org.apache.axis2.description.AxisService;
import org.apache.axis2.description.AxisServiceGroup;
import org.apache.axis2.description.Parameter;
import org.apache.axis2.engine.AxisConfiguration;
import org.wso2.utils.ServerConfiguration;
import org.wso2.wsas.ServerConstants;
import org.wso2.wsas.admin.service.StatisticsClient;
import org.wso2.wsas.util.SystemFilter;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Iterator;

public class ServerStatus {
    private String serverName;
    private String serverStartTime;
    private String systemUpTime;
    private int requestCount;
    private int responseCount;
    private int faultCount;
    private int services;
    private Metric usedMemory;
    private Metric totalMemory;
    private double avgResponseTime;
    private long minResponseTime;
    private long maxResponseTime;
    private String wso2wsasVersion;
    private SimpleDateFormat dateFormatter = new SimpleDateFormat("yyyy-MM-dd' 'HH:mm:ss");
    private String javaVersion;

    private static final long GB_IN_BYTES = 1024 * 1024 * 1024;
    private static final long MB_IN_BYTES = 1024 * 1024;
    private static final long KB_IN_BYTES = 1024;
    private static final int SECONDS_PER_DAY = 3600 * 24;

    private StatisticsClient statClient;

    public ServerStatus() throws AxisFault {
        try {
            statClient = new StatisticsClient();
            javaVersion = System.getProperty("java.version");
        } catch (Exception e) {
            throw AxisFault.makeFault(e);
        }
    }

    public void update(AxisConfiguration axisConfig) throws AxisFault {
        Parameter systemStartTime =
                axisConfig.getParameter("wso2statistics.server.start.time");
        long startTime = ((Long) systemStartTime.getValue()).longValue();
        Date stTime = new Date(startTime);
        serverStartTime = dateFormatter.format(stTime);
        systemUpTime = (getTime((System.currentTimeMillis() - startTime) / 1000));

        try {
            requestCount = statClient.getSystemRequestCount();
            responseCount = statClient.getSystemResponseCount();
            faultCount = statClient.getSystemFaultCount();
            avgResponseTime = statClient.getAvgSystemResponseTime();
            maxResponseTime = statClient.getMaxSystemResponseTime();
            minResponseTime = statClient.getMinSystemResponseTime();
        } catch (Exception e) {
            throw AxisFault.makeFault(e);
        }

        Runtime runtime = Runtime.getRuntime();
        usedMemory = formatMemoryValue(runtime.totalMemory() - runtime.freeMemory());
        totalMemory = formatMemoryValue(runtime.totalMemory());

        wso2wsasVersion = ServerConfiguration.getInstance().getFirstProperty("Version");

        int activeServices = 0;
        for (Iterator services = axisConfig.getServices().values().iterator();
             services.hasNext();) {
            AxisService axisService = (AxisService) services.next();
            AxisServiceGroup asGroup = (AxisServiceGroup) axisService.getParent();
            if (!axisService.isActive() || axisService.isClientSide() ||
                SystemFilter.isFilteredOutService(asGroup.getServiceGroupName())) {
                continue;
            }
            activeServices++;
        }

        this.services = activeServices;
        serverName = ServerConstants.SERVER_NAME;
    }

    public String getSystemUpTime() {
        return systemUpTime;
    }

    public String getServerStartTime() {
        return serverStartTime;
    }

    public String getServerName() {
        return serverName;
    }

    public int getRequestCount() {
        return requestCount;
    }

    public int getServices() {
        return services;
    }

    public Metric getUsedMemory() {
        return usedMemory;
    }

    public Metric getTotalMemory() {
        return totalMemory;
    }

    public int getFaultCount() {
        return faultCount;
    }

    public int getResponseCount() {
        return responseCount;
    }

    private Metric formatMemoryValue(double value) {
        double tempVal = value;
        String unit = "bytes";
        if (value > 0) {
            if ((tempVal = (value / GB_IN_BYTES)) >= 1) {
                tempVal = round(tempVal, 2);
                unit = "GB";
            } else if ((tempVal = (value / MB_IN_BYTES)) >= 1) {
                tempVal = round(tempVal, 2);
                unit = " MB";
            } else if ((tempVal = (value / KB_IN_BYTES)) >= 1) {
                tempVal = round(tempVal, 2);
                unit = " KB";
            }
        }
        return new Metric(tempVal, unit);
    }

    private double round(double value, int numOfDecimalPlaces) {
        long val = 1;
        for (int i = 1; i <= numOfDecimalPlaces; i++) {
            val *= 10;
        }
        return Math.round(value * val) / (double) val;
    }

    private String getTime(long timeInSeconds) {
        long days;
        long hours;
        long minutes;
        long seconds;
        days = timeInSeconds / SECONDS_PER_DAY;
        timeInSeconds = timeInSeconds - (days * SECONDS_PER_DAY);
        hours = timeInSeconds / 3600;
        timeInSeconds = timeInSeconds - (hours * 3600);
        minutes = timeInSeconds / 60;
        timeInSeconds = timeInSeconds - (minutes * 60);
        seconds = timeInSeconds;

        return days + " day(s) " + hours + " hr(s) " + minutes + " min(s) " + seconds + " sec(s)";
    }

    public double getAvgResponseTime() {
        return avgResponseTime;
    }

    public long getMinResponseTime() {
        return minResponseTime;
    }

    public long getMaxResponseTime() {
        return maxResponseTime;
    }

    public String getWso2wsasVersion() {
        return wso2wsasVersion;
    }

    public void setWso2wsasVersion(String wso2wsasVersion) {
        this.wso2wsasVersion = wso2wsasVersion;
    }

    public String getJavaVersion() {
        return javaVersion;
    }

    public void setJavaVersion(String javaVersion) {
        this.javaVersion = javaVersion;
    }
}
