/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.admin.service.logging;

import org.apache.axis2.AxisFault;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.log4j.Appender;
import org.apache.log4j.AppenderSkeleton;
import org.apache.log4j.FileAppender;
import org.apache.log4j.Layout;
import org.apache.log4j.Level;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import org.apache.log4j.PatternLayout;
import org.apache.log4j.Priority;
import org.apache.log4j.net.SyslogAppender;
import org.wso2.wsas.ServerConstants;
import org.wso2.wsas.admin.service.util.AppenderData;
import org.wso2.wsas.admin.service.util.LogData;
import org.wso2.wsas.admin.service.util.LoggerData;
import org.wso2.wsas.persistence.PersistenceManager;
import org.wso2.wsas.persistence.dataobject.AppenderDO;
import org.wso2.wsas.persistence.dataobject.LoggerDO;
import org.wso2.wsas.util.LoggingUtil;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * This is the Admin service used for obtaining Log4J information about the system and also used for
 * managing the system Log4J configuration
 */
public class LoggingAdmin {
    private static final Log log = LogFactory.getLog(LoggingAdmin.class);

    private PersistenceManager pm = new PersistenceManager();

    /**
     * Globally update the System Logging configuration. The global logging level & the log pattern
     * will be changed by this method
     *
     * @param logLevel   The global log level to be set
     * @param logPattern The global log pattern to be set
     * @param persist    true - indicates persist these changes to the DB; false - indicates make
     *                   changes only in memory and do not persist the changes to DB
     */
    public void updateSystemLog(String logLevel, String logPattern, boolean persist) {

        Set appenderSet = new HashSet();

        // update root logger details
        Logger rootLogger = Logger.getRootLogger();
        rootLogger.setLevel(Level.toLevel(logLevel));
        if (persist) {
            pm.updateConfigurationProperty(ServerConstants.Logging.SYSTEM_LOG_LEVEL,
                                           logLevel);
            pm.updateConfigurationProperty(ServerConstants.Logging.SYSTEM_LOG_PATTERN,
                                           logPattern);
        }
        addAppendersToSet(rootLogger.getAllAppenders(), appenderSet);

        // update logger and appender data, following are set
        // 1. log level of all the loggers to logLevel
        // 2. pattern of all the appenders to logpattern
        Enumeration loggers = LogManager.getCurrentLoggers();
        Logger logger;
        LoggerDO loggerDO;

        while (loggers.hasMoreElements()) {
            logger = (Logger) loggers.nextElement();
            // we ignore all class level defined loggers
            if (logger.getLevel() != null) {
                addAppendersToSet(logger.getAllAppenders(), appenderSet);

                if ((logLevel != null) && (logLevel.trim().length() != 0)) {
                    loggerDO = pm.getLoggerDO(logger.getName());
                    logger.setLevel(Level.toLevel(logLevel));
                    if (persist) {
                        pm.updateLoggerData(loggerDO.getName(), logLevel,
                                            loggerDO.getAdditivity());
                    }
                }
            }
        }

        // update the appender data according to data stored in database
        Appender appender;
        AppenderDO appenderDO;

        for (Iterator iter = appenderSet.iterator(); iter.hasNext();) {
            appender = (Appender) iter.next();
            appenderDO = pm.getAppenderDO(appender.getName());

            if ((appender.getLayout() != null) &&
                (appender.getLayout() instanceof PatternLayout)) {
                if ((logPattern != null) && (logPattern.trim().length() != 0)) {
                    ((PatternLayout) appender.getLayout()).setConversionPattern(logPattern);
                    if (persist) {
                        pm.updateAppenderData(appenderDO.getName(),
                                              logPattern,
                                              logLevel,
                                              appenderDO.getLogFileName(),
                                              appenderDO.getSysLogHost(),
                                              appenderDO.getFacility(),
                                              appenderDO.getIsFileAppender(),
                                              appenderDO.getIsSysLogAppender());
                    }
                }

                // set the threshold
                if (appender instanceof AppenderSkeleton) {
                    AppenderSkeleton appenderSkeleton = (AppenderSkeleton) appender;
                    appenderSkeleton.setThreshold(Level.toLevel(logLevel));
                    appenderSkeleton.activateOptions();
                }
            }
        }
        LoggingUtil.setSystemLoggingParameters(logLevel, logPattern);
    }

    /**
     * @param appenderName The name of the appender
     * @return The appender information the given appender with name <code>appenderName</code>
     */
    public AppenderData getAppenderData(String appenderName) {
        Logger rootLogger = Logger.getRootLogger();
        Appender targetAppender = getAppenderInLoggerWithName(rootLogger, appenderName);
        if (targetAppender == null) {
            Enumeration loggers = LogManager.getCurrentLoggers();
            while (loggers.hasMoreElements()) {
                Logger logger = (Logger) loggers.nextElement();
                targetAppender = getAppenderInLoggerWithName(logger, appenderName);
                if (targetAppender != null) {
                    break;
                }
            }
        }
        return toAppenderData(targetAppender);
    }

    /**
     * Convert a Log$J Appender to an instance of {@link AppenderData}
     *
     * @param targetAppender The Appender to be converted
     * @return The {@link AppenderData} instance corresponding to <code>targetAppender</code>
     */
    private AppenderData toAppenderData(Appender targetAppender) {
        AppenderData appenderData = null;
        if (targetAppender != null) {
            appenderData = new AppenderData();
            appenderData.setName(targetAppender.getName());
            Layout layout = targetAppender.getLayout();
            if (layout instanceof PatternLayout) {
                appenderData.setPattern(((PatternLayout) layout).getConversionPattern());
            }
            if (targetAppender instanceof AppenderSkeleton) {          // normally all the appenders inherit from AppenderSkelton
                AppenderSkeleton appender = (AppenderSkeleton) targetAppender;
                Priority priority = appender.getThreshold();
                if (priority != null) {
                    appenderData.setThreshold(priority.toString());
                } else {
                    appender.setThreshold(Level.toLevel(Priority.DEBUG_INT));
                    appenderData.setThreshold("DEBUG");
                }
            }
            if (targetAppender instanceof SyslogAppender) { //NOTE: Don't make this an else if
                SyslogAppender appender = (SyslogAppender) targetAppender;
                appenderData.setIsSysLogAppender(true);
                appenderData.setFacility(appender.getFacility());
                appenderData.setSysLogHost(appender.getSyslogHost());
            } else if (targetAppender instanceof FileAppender) {
                appenderData.setIsFileAppender(true);
                appenderData.setLogFile(((FileAppender) targetAppender).getFile());
            }
        }
        return appenderData;
    }

    private Appender getAppenderInLoggerWithName(Logger logger, String appenderName) {
        Enumeration appenders = logger.getAllAppenders();
        Appender targetAppender = null;
        while (appenders.hasMoreElements()) {
            Appender appender = (Appender) appenders.nextElement();
            if (appender.getName().equals(appenderName)) {
                targetAppender = appender;
                break;
            }
        }
        return targetAppender;
    }

    public LoggerData getLoggerData(String loggerName) {

        Logger logger = LogManager.getLogger(loggerName);
        String parentName =
                (logger.getParent() == null ? "empty" : logger.getParent().getName());
        return new LoggerData(logger.getName(),
                              logger.getEffectiveLevel().toString(),
                              logger.getAdditivity(),
                              parentName);
    }

    /**
     * Set the Appender information. We receive all the parameters from the update appenders method
     * but we have to only update the relevent data.
     *
     * @param appenderName    The name of the Appender
     * @param appenderPattern The log pattern
     * @param threshold       The logging threshold
     * @param logFileName     log file name - Only relevant to FileAppenders
     * @param sysLogHost      The Syslog host - Only relevant to SyslogAppenders
     * @param facility        The Syslog facility - Only relevant to SyslogAppenders
     * @param persist         true - indicates persist these changes to the DB; false - indicates
     *                        make changes only in memory and do not persist the changes to DB
     * @throws AxisFault If failure occurs during setting of these values
     */
    public void updateAllAppenderData(String appenderName,
                                      String appenderPattern,
                                      String threshold,
                                      String logFileName,
                                      String sysLogHost,
                                      String facility,
                                      boolean persist) throws AxisFault {

        // get the existing appender and set the is File appender value
        AppenderDO appenderDO = pm.getAppenderDO(appenderName);

        if (appenderDO.getIsFileAppender()) {
            // Check if the file is valid
            logFileName = logFileName.replace('\\', '/');
            File logFile = new File(logFileName);
            if (!logFile.isAbsolute()) {
                if (logFileName.startsWith("./")) {
                    logFileName = logFileName.substring(2);
                }
                logFileName = (System.getProperty(ServerConstants.WSO2WSAS_HOME) + "/" +
                               logFileName).replace('\\', '/');
                logFile = new File(logFileName);
            }
            if (!logFile.exists()) {
                int lastIndex = logFileName.lastIndexOf("/");
                String msg = "Cannot create logfile " + logFileName +
                             ". Please verify that the logging directory exists, log file name is " +
                             "valid and that you have read-write access to this file.";
                if (lastIndex != -1) {
                    String dirName = logFileName.substring(0, lastIndex);
                    File dir = new File(dirName);
                    if (!dir.exists() && !dir.mkdirs()) {
                        throw new AxisFault(msg);
                    }
                }
                try {
                    if (!logFile.createNewFile()) {
                        throw new AxisFault(msg);
                    }
                } catch (IOException e) {
                    throw new AxisFault(msg);
                }
            }
            if (persist) {
                pm.updateAppenderData(appenderName, appenderPattern, threshold,
                                      logFileName, null, null, true, false);
            }
        } else if (appenderDO.getIsSysLogAppender()) {
            if (persist) {
                pm.updateAppenderData(appenderName, appenderPattern, threshold,
                                      null, sysLogHost, facility, false, true);
            }
        } else {
            if (persist) {
                pm.updateAppenderData(appenderName, appenderPattern, threshold,
                                      null, null, null, false, false);
            }
        }

        // update system appender data
        Set appenderSet = new HashSet();
        Logger rootLogger = Logger.getRootLogger();
        addAppendersToSet(rootLogger.getAllAppenders(), appenderSet);

        Enumeration loggers = LogManager.getCurrentLoggers();
        while (loggers.hasMoreElements()) {
            Logger logger = (Logger) loggers.nextElement();
            if (logger.getLevel() != null) {
                addAppendersToSet(logger.getAllAppenders(), appenderSet);
            }
        }
        Appender appender = null;
        for (Iterator iter = appenderSet.iterator(); iter.hasNext();) {
            appender = (Appender) iter.next();
            if (appender.getName().equals(appenderName)) {
                break;
            }
        }

        if (appender != null) {
            if ((appender.getLayout() != null) &&
                (appender.getLayout() instanceof PatternLayout)) {
                ((PatternLayout) appender.getLayout()).setConversionPattern(appenderPattern);
            }

            if (appender instanceof FileAppender) {
                ((FileAppender) appender).setFile(logFileName);
                if (log.isDebugEnabled()) {
                    log.debug("change the logfile of the appender ==> " +
                              appender.getName() + " to " + logFileName);
                }
                ((FileAppender) appender).activateOptions();
            }

            if (appender instanceof SyslogAppender) {
                SyslogAppender syslogAppender = (SyslogAppender) appender;
                syslogAppender.setSyslogHost(sysLogHost);
                syslogAppender.setFacility(facility);
            }

            // set the threshold
            if (appender instanceof AppenderSkeleton) {
                AppenderSkeleton appenderSkeleton = (AppenderSkeleton) appender;
                appenderSkeleton.setThreshold(Level.toLevel(threshold));
                appenderSkeleton.activateOptions();
            }
        }
    }

    public void updateLoggerData(String loggerName,
                                 String loggerLevel,
                                 boolean additivity,
                                 boolean persist) {

        if (persist) {
            pm.updateLoggerData(loggerName, loggerLevel, additivity);
        }

        //update logger data in current system
        Logger logger = LogManager.getLogger(loggerName);

        if (logger != null) {
            logger.setLevel(Level.toLevel(loggerLevel));
            logger.setAdditivity(additivity);
            if (log.isDebugEnabled()) {
                log.debug("Set the log level of logger ==>" + logger.getName() +
                          " to " + logger.getLevel().toString());
            }
        }
    }

    /**
     * Get all the information regarding the system log4j information but Logger information
     * such as logger, parent logger, effective level and additivity. As this information is
     * abundant, rendering SOAP env will be expensive and time consuming.
     *
     * @return System log information
     */
    public LogData getSystemLog() {
        LogData logData = new LogData();

        // loading initial data
        logData.setLogLevel(LoggingUtil.getSystemLogLevel());
        logData.setLogPattern(LoggingUtil.getSystemLogPattern());

        // set the appenders
        AppenderData[] appenderData = getAllAppenderData();
        if (appenderData.length > 0) {
            logData.setAppenderData(appenderData);
            logData.setSelectedAppenderData(appenderData[0]);
        }

        // set the loggers
        /*LoggerData[] loggerData = getAllLoggerData(null, false);
        if (loggerData.length > 0) {
            logData.setLoggerData(loggerData);
            logData.setSelectedLoggerData(loggerData[0]);
        }*/

        return logData;
    }

    public void restoreDefaults() throws AxisFault {
        LoggingUtil.restoreDefaults();
        LoggingUtil.loadDefaultConfiguration();
    }

    private AppenderData[] getAllAppenderData() {
        Set appenderSet = new HashSet();
        Logger rootLogger = Logger.getRootLogger();
        Enumeration appenders = rootLogger.getAllAppenders();
        addAppendersToSet(appenders, appenderSet);

        Enumeration loggers = LogManager.getCurrentLoggers();
        while (loggers.hasMoreElements()) {
            Logger logger = (Logger) loggers.nextElement();
            addAppendersToSet(logger.getAllAppenders(), appenderSet);
        }
        AppenderData[] appenderDataArray = new AppenderData[appenderSet.size()];
        int i = 0;
        for (Iterator iterator = appenderSet.iterator(); iterator.hasNext();) {
            appenderDataArray[i] = toAppenderData((Appender) iterator.next());
            i++;
        }
        Arrays.sort(appenderDataArray,
                    new Comparator() {
                        public int compare(Object arg0, Object arg1) {
                            AppenderData a = (AppenderData) arg0;
                            AppenderData b = (AppenderData) arg1;
                            return a.getName().compareTo(b.getName());
                        }
                    });
        return appenderDataArray;
    }

    public LoggerData[] getAllLoggerData(String logNameFilter, boolean beginsWith) {
        Enumeration loggers = LogManager.getCurrentLoggers();
        List list = new ArrayList();
        while (loggers.hasMoreElements()) {
            Logger logger = (Logger) loggers.nextElement();
            if ((logNameFilter != null && beginsWith && logger.getName().startsWith(logNameFilter)) || // Logger name begins with logNameFilter
                (logNameFilter != null && !beginsWith && logger.getName().indexOf(logNameFilter) != -1) || // Logger name contains logNameFilter
                (logNameFilter == null || logNameFilter.trim().length() == 0)) {  // No logNameFilter specified
                String parentName =
                        (logger.getParent() == null ? "-" : logger.getParent().getName());
                LoggerData loggerData = new LoggerData(logger.getName(),
                                                       logger.getEffectiveLevel().toString(),
                                                       logger.getAdditivity(),
                                                       parentName);
                list.add(loggerData);
            }
        }
        Collections.sort(list,
                         new Comparator() {
                             public int compare(Object arg0, Object arg1) {
                                 LoggerData a = (LoggerData) arg0;
                                 LoggerData b = (LoggerData) arg1;
                                 return a.getName().compareTo(b.getName());
                             }
                         });
        Logger rootLogger = LogManager.getRootLogger();
        if ((logNameFilter != null && beginsWith && rootLogger.getName().startsWith(logNameFilter)) || // Logger name begins with logNameFilter
            (logNameFilter != null && !beginsWith && rootLogger.getName().indexOf(logNameFilter) != -1) || // Logger name contains logNameFilter
            (logNameFilter == null || logNameFilter.trim().length() == 0)) {  // No logNameFilter specified
            LoggerData loggerData = new LoggerData(rootLogger.getName(),
                                                   rootLogger.getEffectiveLevel().toString(),
                                                   rootLogger.getAdditivity(),
                                                   "-");
            list.add(0, loggerData);
        }
        return (LoggerData[]) list.toArray(new LoggerData[list.size()]);
    }

    private void addAppendersToSet(Enumeration appenders, Set appenderSet) {
        Appender appender;
        while (appenders.hasMoreElements()) {
            appender = (Appender) appenders.nextElement();
            appenderSet.add(appender);
            if (log.isDebugEnabled()) {
                log.debug("Add appender ==> " + appender.getName() + " to appender set");
            }
        }
    }
}
