/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.admin.service;

import org.apache.axis2.AxisFault;
import org.wso2.utils.AbstractAdmin;
import org.wso2.utils.ServerConfiguration;
import org.wso2.utils.ServerException;
import org.wso2.utils.security.CryptoException;
import org.wso2.utils.security.CryptoUtil;
import org.wso2.wsas.ServerConstants;
import org.wso2.wsas.admin.service.util.RoleData;
import org.wso2.wsas.admin.service.util.UserData;
import org.wso2.wsas.persistence.PersistenceManager;
import org.wso2.wsas.persistence.dataobject.ServiceUserDO;
import org.wso2.wsas.persistence.dataobject.ServiceUserRoleDO;
import org.wso2.wsas.persistence.exception.ServiceUserAlreadyExistsException;
import org.wso2.wsas.persistence.exception.ServiceUserNotFoundException;
import org.wso2.wsas.persistence.exception.UserRoleAlreadyExistsException;

import java.io.File;


/**
 * Admin service to manage WSO2 WSAS users
 */
public class UserAdmin extends AbstractAdmin {
    private PersistenceManager pm;

    public UserAdmin() {
        pm = new PersistenceManager();
    }

    public void addAdmin(String userName, String password)
            throws AxisFault {
        try {
            if ((userName == null) || (userName.trim().length() == 0)) {
                throw new AxisFault("Username cannot be null or empty");
            }

            if ((password == null) || (password.trim().length() == 0)) {
                throw new AxisFault("Password cannot be null or empty");
            }

            ServiceUserDO admin = new ServiceUserDO();
            admin.setUsername(userName);
            ServerConfiguration config = ServerConfiguration.getInstance();
            CryptoUtil cryptoUtil =
                    new CryptoUtil(new File(config.getFirstProperty("Security.KeyStore.Location")).getAbsolutePath(),
                                   config.getFirstProperty("Security.KeyStore.Password"),
                                   config.getFirstProperty("Security.KeyStore.KeyAlias"),
                                   config.getFirstProperty("Security.KeyStore.KeyPassword"),
                                   config.getFirstProperty("Security.KeyStore.Type"));
            admin.setPassword(cryptoUtil.encryptAndBase64Encode(password.getBytes()));

            ServiceUserRoleDO userRole = new ServiceUserRoleDO();
            userRole.setRole(ServerConstants.ADMIN_ROLE);
            pm.addUser(admin);
        } catch (CryptoException e) {
            throw AxisFault.makeFault(e);
        } catch (ServiceUserAlreadyExistsException e) {
            throw new AxisFault("Admin with username " + userName +
                                " already exists");
        }
    }

    public void editAdmin(String userName, String password)
            throws AxisFault {
        try {
            if ((userName == null) || (userName.trim().length() == 0)) {
                throw new AxisFault("Username cannot be null or empty");
            }

            if ((password == null) || (password.trim().length() == 0)) {
                throw new AxisFault("Password cannot be null or empty");
            }

            ServiceUserDO admin = new ServiceUserDO();
            admin.setUsername(userName);
            ServerConfiguration config = ServerConfiguration.getInstance();
            CryptoUtil cryptoUtil =
                    new CryptoUtil(new File(config.getFirstProperty("Security.KeyStore.Location")).getAbsolutePath(),
                                   config.getFirstProperty("Security.KeyStore.Password"),
                                   config.getFirstProperty("Security.KeyStore.KeyAlias"),
                                   config.getFirstProperty("Security.KeyStore.KeyPassword"),
                                   config.getFirstProperty("Security.KeyStore.Type"));
            admin.setPassword(cryptoUtil.encryptAndBase64Encode(password.getBytes()));

            ServiceUserRoleDO userRole = new ServiceUserRoleDO();
            userRole.setRole(ServerConstants.ADMIN_ROLE);

            pm.updateUser(admin);
        } catch (Exception e) {
            throw AxisFault.makeFault(e);
        }
    }

    public boolean editUserPassword(String oldPassword, String username,
                                    String password) throws Exception {
        if ((username == null) || (username.trim().length() == 0)) {
            return false;
        }

        if ((password == null) || (password.trim().length() == 0)) {
            return false;
        }

        if ((oldPassword == null) || (oldPassword.trim().length() == 0)) {
            return false;
        }

        if (username.equals("admin") && password.equals("admin")) {
            throw new Exception("This password is not allowed for Administrator 'admin'");
        }

        ServiceUserDO serviceUserDO = pm.getUser(username);

        if (serviceUserDO == null) {
            return false;
        }

        ServerConfiguration config = ServerConfiguration.getInstance();
        CryptoUtil cryptoUtil =
                new CryptoUtil(new File(config.getFirstProperty("Security.KeyStore.Location")).getAbsolutePath(),
                               config.getFirstProperty("Security.KeyStore.Password"),
                               config.getFirstProperty("Security.KeyStore.KeyAlias"),
                               config.getFirstProperty("Security.KeyStore.KeyPassword"),
                               config.getFirstProperty("Security.KeyStore.Type"));
        String origOldPwd = new String(cryptoUtil.base64DecodeAndDecrypt(
                serviceUserDO.getPassword()));

        if (!origOldPwd.equals(oldPassword)) {
            return false;
        }

        serviceUserDO.setPassword(cryptoUtil.encryptAndBase64Encode(password.getBytes()));
        pm.updateUser(serviceUserDO);

        return true;
    }

    public boolean changeUserPassword(String username, String password) throws Exception {
        if ((username == null) || (username.trim().length() == 0)) {
            return false;
        }

        if ((password == null) || (password.trim().length() == 0)) {
            return false;
        }

        if (username.equals("admin") && password.equals("admin")) {
            throw new Exception("This password is not allowed for Administrator 'admin'");
        }

        ServiceUserDO serviceUserDO = pm.getUser(username);

        if (serviceUserDO == null) {
            return false;
        }

        ServerConfiguration config = ServerConfiguration.getInstance();
        CryptoUtil cryptoUtil =
                new CryptoUtil(new File(config.getFirstProperty("Security.KeyStore.Location")).getAbsolutePath(),
                               config.getFirstProperty("Security.KeyStore.Password"),
                               config.getFirstProperty("Security.KeyStore.KeyAlias"),
                               config.getFirstProperty("Security.KeyStore.KeyPassword"),
                               config.getFirstProperty("Security.KeyStore.Type"));
        serviceUserDO.setPassword(cryptoUtil.encryptAndBase64Encode(password.getBytes()));
        pm.updateUser(serviceUserDO);

        return true;
    }

    public boolean editUserRole(String username, String role)
            throws Exception {
        if ((username == null) || (username.trim().length() == 0)) {
            return false;
        }

        ServiceUserDO serviceUserDO = pm.getUser(username);

        if (serviceUserDO == null) {
            return false;
        }

        pm.updateUser(serviceUserDO);

        return true;
    }

    public void removeUser(String username) throws AxisFault {
        if ((username == null) || (username.trim().length() == 0)) {
            throw new AxisFault("Username cannot be null or empty");
        }

        ServiceUserDO serviceUserDO = pm.getUser(username);

        if (serviceUserDO == null) {
            throw new AxisFault("Invalid user , does not exist in the system" +
                                username);
        }

        try {
            pm.removeUser(username);
        } catch (ServiceUserNotFoundException e) {
            throw new AxisFault(e.getMessage());
        }
    }

    /**
     * Assign a role to an existing user
     *
     * @param username The username of the user to whom the role is to be assigned
     * @param role     The role to be assigned
     * @throws AxisFault If role assignment fails
     */
    public void assignRoleToUser(String username, String role) throws AxisFault {
        try {
            ServiceUserDO serviceUserDO = pm.getUser(username);

            if (serviceUserDO != null) {
                ServiceUserRoleDO roleDO = pm.getUserRole(role);

                if (serviceUserDO.getRoles().contains(roleDO)) {
                    throw new AxisFault("User \'" + username + "\' already has role \'" +
                                        role + "\'.");
                }
                pm.addRole(username, roleDO);
            }
        } catch (Exception e) {
            throw AxisFault.makeFault(e);
        }
    }

    public String addUser(String username, String password, String role,
                          String description) throws AxisFault {
        if ((username == null) || (username.trim().length() == 0)) {
            return "Username cannot be null or empty";
        }

        if ((password == null) || (password.trim().length() == 0)) {
            return "Password cannot be null or empty";
        }

        if ((role == null) || (role.trim().length() == 0)) {
            return "User Role cannot be null or empty";
        }

        try {
            ServiceUserDO serviceUserDO = new ServiceUserDO();
            serviceUserDO.setUsername(username);
            ServerConfiguration config = ServerConfiguration.getInstance();
            CryptoUtil cryptoUtil =
                    new CryptoUtil(new File(config.getFirstProperty("Security.KeyStore.Location")).getAbsolutePath(),
                                   config.getFirstProperty("Security.KeyStore.Password"),
                                   config.getFirstProperty("Security.KeyStore.KeyAlias"),
                                   config.getFirstProperty("Security.KeyStore.KeyPassword"),
                                   config.getFirstProperty("Security.KeyStore.Type"));
            serviceUserDO.setPassword(cryptoUtil.encryptAndBase64Encode(
                    password.getBytes()));
            serviceUserDO.setDescription(description);
            pm.addUser(serviceUserDO);

            ServiceUserRoleDO roleDO = pm.getUserRole(role);
            pm.addRole(username, roleDO);
        } catch (ServiceUserAlreadyExistsException e) {
            return "User with username " + username + " already exists!";
        } catch (CryptoException e) {
            throw new AxisFault("Exception Occurred", e);
        }

        return "User with username " + username + " successfully added";
    }

    public UserData[] getUserNames() throws AxisFault {
        ServiceUserDO[] users = pm.getUsers();

        if ((users == null) || (users.length == 0)) {
            return new UserData[0];
        }

        ServiceUserDO serviceUser;
        UserData[] udata = new UserData[users.length];

        for (int i = 0; i < users.length; i++) {
            serviceUser = users[i];

            ServiceUserRoleDO[] roles = pm.getUserSpecificRoles(serviceUser.getUsername());
            RoleData[] roleData = new RoleData[roles.length];
            ServiceUserRoleDO serviceRole;

            for (int j = 0; j < roleData.length; j++) {
                serviceRole = roles[j];

                RoleData data = new RoleData();
                data.setRole(serviceRole.getRole());
                data.setDescription(serviceRole.getDescription());
                roleData[j] = data; // role datat done
            }

            UserData userdata = new UserData();
            userdata.setRoles(roleData);
            userdata.setUserName(serviceUser.getUsername());
            userdata.setDescription(serviceUser.getDescription());
            udata[i] = userdata;
        }

        return udata;
    }

    public String[] getUsers() throws AxisFault {
        ServiceUserDO[] users = pm.getUsers();

        if ((users == null) || (users.length == 0)) {
            return new String[0];
        }

        String[] udata = new String[users.length];

        for (int i = 0; i < users.length; i++) {
            udata[i] = users[i].getUsername();
        }

        return udata;
    }

    public boolean addRole(String role, String description)
            throws AxisFault {
        if ((role == null) || (role.trim().length() == 0)) {
            return false;
        }

        ServiceUserRoleDO serviceUserRole = new ServiceUserRoleDO();
        serviceUserRole.setRole(role);
        serviceUserRole.setDescription(description);

        try {
            pm.addUserRole(serviceUserRole);
        } catch (UserRoleAlreadyExistsException e) {
            return false;
        }
        return true;
    }

    public String[] getRoleNames() throws AxisFault {
        ServiceUserRoleDO[] userRoles = pm.getUserRoles();

        if ((userRoles == null) || (userRoles.length == 0)) {
            return new String[0];
        }

        String[] roles = new String[userRoles.length];

        for (int i = 0; i < userRoles.length; i++) {
            roles[i] = userRoles[i].getRole();
        }

        return roles;
    }

    public RoleData[] getRoleNamesAndDescriptions() throws AxisFault {
        ServiceUserRoleDO[] userRoles = pm.getUserRoles();

        if ((userRoles == null) || (userRoles.length == 0)) {
            return new RoleData[0];
        }

        ServiceUserRoleDO serviceUserRole;
        RoleData[] rdata = new RoleData[userRoles.length];

        for (int i = 0; i < rdata.length; i++) {
            serviceUserRole = userRoles[i];

            RoleData roleData = new RoleData();
            roleData.setRole(serviceUserRole.getRole());
            roleData.setDescription(serviceUserRole.getDescription());
            rdata[i] = roleData;
        }

        return rdata;
    }

    public void deleteUser(String username) throws AxisFault {
        pm.deleteUser(username);
    }

    public String deleteRoleCompletely(String role) throws AxisFault {
        try {
            ServiceUserDO[] userDOs = pm.getUsers();

            //Need not to check null because admin is a trivial super user
            StringBuffer usersWithRole = new StringBuffer();

            for (int i = 0; i < userDOs.length; i++) {
                ServiceUserDO userDO = userDOs[i];
                ServiceUserRoleDO[] roleDOs = pm.getUserSpecificRoles(userDO.getUsername());

                if (roleDOs.length == 1) {
                    if (roleDOs[0].getRole().equalsIgnoreCase(role.trim())) {
                        usersWithRole.append(userDO.getUsername()).append(",");
                    }
                }
            }

            if (usersWithRole.length() != 0) {
                return getDeleteMsg(false, usersWithRole.toString(), role);
            }
        } catch (Exception e) {
            throw AxisFault.makeFault(e);
        }

        pm.deleteRole(role);

        return getDeleteMsg(true, null, role);
    }

    private String getDeleteMsg(boolean canDelete, String users, String role) {
        String cannotDeleteStatement = "Role '" + role +
                                       "' cannot be deleted.\n" + "since it is associated with users; " +
                                       users + " having only this role.\n" +
                                       "Assign these users with a different role before trying to " +
                                       "delete this role.";
        String canDeleteStatement = "Role '" + role + "' deleted successfully.";

        if (canDelete) {
            return canDeleteStatement;
        } else {
            return cannotDeleteStatement;
        }
    }

    public boolean deleteRoleFromUser(String username, String role)
            throws AxisFault, ServerException {
        if ((username == null) || (username.trim().length() == 0)) {
            throw new ServerException("Username is invalid");
        }

        if ((role == null) || (role.trim().length() == 0)) {
            throw new ServerException("Role is invalid");
        }

        ServiceUserDO serviceUserDO = pm.getUser(username);

        if (serviceUserDO == null) {
            throw new ServerException("ServiceUser is invalid" + username);
        }

        ServiceUserRoleDO[] roles = pm.getUserSpecificRoles(username);

        if (roles.length == 1) {
            return false;
        } else {
            pm.deleteRoleFromUser(username, role);

            return true;
        }
    }
}
