/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.admin.service;

import org.apache.axiom.om.OMElement;
import org.apache.axis2.AxisFault;
import org.apache.axis2.Constants;
import org.apache.axis2.description.AxisOperation;
import org.apache.axis2.description.AxisService;
import org.apache.axis2.description.AxisServiceGroup;
import org.apache.axis2.description.Parameter;
import org.apache.axis2.description.PolicyInclude;
import org.apache.axis2.engine.AxisConfiguration;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.neethi.Policy;
import org.wso2.utils.AbstractAdmin;
import org.wso2.utils.FileManipulator;
import org.wso2.utils.ServerConfiguration;
import org.wso2.wsas.ServerConstants;
import org.wso2.wsas.admin.service.util.ServiceArchiveCreator;
import org.wso2.wsas.admin.service.util.ServiceGroupMetaData;
import org.wso2.wsas.persistence.PersistenceManager;
import org.wso2.wsas.persistence.dataobject.OperationDO;
import org.wso2.wsas.persistence.dataobject.OperationParameterDO;
import org.wso2.wsas.persistence.dataobject.ServiceDO;
import org.wso2.wsas.persistence.dataobject.ServiceGroupDO;
import org.wso2.wsas.persistence.dataobject.ServiceGroupParameterDO;
import org.wso2.wsas.persistence.dataobject.ServiceParameterDO;
import org.wso2.wsas.persistence.exception.DuplicateEntityException;
import org.wso2.wsas.util.ParameterUtil;
import org.wso2.wsas.util.PolicyUtil;
import org.wso2.wsas.util.SystemFilter;
import org.wso2.wsas.util.WsasUtils;

import java.io.File;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;

/**
 * Admin service to manage service groups
 */
public class ServiceGroupAdmin extends AbstractAdmin {
    private static Log log = LogFactory.getLog(ServiceGroupAdmin.class);

    private PersistenceManager pm = new PersistenceManager();

    /**
     * Return service group details for a given service group
     *
     * @param serviceGroupName
     * @return ServiceGroupMetaData
     * @throws AxisFault
     */
    public ServiceGroupMetaData listServiceGroup(String serviceGroupName) throws AxisFault {
        ServiceGroupMetaData sgmd = new ServiceGroupMetaData();

        AxisServiceGroup serviceGroup = getAxisConfig().getServiceGroup(serviceGroupName);
        sgmd.setGroup_name(serviceGroup.getServiceGroupName());


        Parameter parameter = serviceGroup.getParameter(Constants.Configuration.ENABLE_MTOM);
        if (parameter != null) {
            sgmd.setEnableMTOM((String) parameter.getValue());
        }

        return sgmd;

    }

    /**
     * set the service group parameter enableMTOM to manipulate MTOM flag true/false/optional
     *
     * @param flag
     * @return ServiceGroupMetaData
     */
    public ServiceGroupMetaData configureMTOM(String flag,
                                              String serviceGroupName) throws AxisFault {
        AxisServiceGroup serviceGroup = getAxisConfig().getServiceGroup(serviceGroupName);
        if (serviceGroup == null) {
            throw new AxisFault("Service group " + serviceGroupName + "cannnot be found!");
        }

        // get the declared parameters
        ArrayList parameters = serviceGroup.getParameters();

        boolean found = false;
        for (Iterator iterator = parameters.iterator(); iterator.hasNext();) {
            Parameter parameter = (Parameter) iterator.next();

            if (parameter.getParameterType() == Parameter.TEXT_PARAMETER &&
                parameter.getValue().toString().equals(Constants.Configuration.ENABLE_MTOM)) {
                parameter.setValue(flag.trim());
                found = true;
                break;
            }

        }
        if (!found) {
            Parameter parameter = ParameterUtil
                    .createParameter(Constants.Configuration.ENABLE_MTOM, flag.trim());
            serviceGroup.addParameter(parameter);
        }

        Parameter parameter = serviceGroup.getParameter(Constants.Configuration.ENABLE_MTOM);

        //set it in all underline childern servies and operations
        for (Iterator iterator = serviceGroup.getServices(); iterator.hasNext();) {
            AxisService axisServce = (AxisService) iterator.next();
            axisServce.addParameter(ParameterUtil.createParameter(
                    Constants.Configuration.ENABLE_MTOM, (String) parameter.getValue()));
            for (Iterator iterator1 = axisServce.getOperations(); iterator1.hasNext();) {
                AxisOperation axisOperation = (AxisOperation) iterator1.next();
                axisOperation.addParameter(ParameterUtil.createParameter(
                        Constants.Configuration.ENABLE_MTOM, (String) parameter.getValue()));
            }
        }

        ServiceGroupDO serviceGroupDO = pm.getServiceGroup(serviceGroupName);
        ServiceGroupParameterDO sgParamDO = serviceGroupDO.getParameter(parameter.getName());

        if (sgParamDO != null) {
            sgParamDO.setValue(parameter.getParameterElement().toString());
            pm.updateEntity(sgParamDO);
        } else {
            sgParamDO = new ServiceGroupParameterDO();
            sgParamDO.setName(parameter.getName());
            sgParamDO.setValue(parameter.getParameterElement().toString());
            sgParamDO.setServiceGroup(serviceGroupDO);
            try {
                pm.addEntity(sgParamDO);
            } catch (DuplicateEntityException e) {
                log.error("Service Parameter already exists", e);
            }
        }

        serviceGroupDO = pm.getServiceGroup(serviceGroupName);
        if (serviceGroupDO != null) {
            ServiceDO[] serviceDOs = pm.getServices(serviceGroupDO);
            if (serviceDOs != null) {
                for (int i = 0; i < serviceDOs.length; i++) {
                    ServiceDO serviceDO = serviceDOs[i];
                    ServiceParameterDO serviceParameterDO =
                            serviceDO.getParameter(parameter.getName());
                    if (serviceParameterDO != null) {
                        serviceParameterDO.setValue(parameter.getParameterElement().toString());
                        pm.updateEntity(serviceParameterDO);
                    } else {
                        serviceParameterDO = new ServiceParameterDO();
                        serviceParameterDO.setName(parameter.getName());
                        serviceParameterDO.setValue(parameter.getParameterElement().toString());
                        serviceParameterDO.setService(serviceDO);
                        try {
                            pm.addEntity(serviceParameterDO);
                        } catch (DuplicateEntityException e) {
                            log.error("Service Parameter already exists", e);
                        }
                    }

                    OperationDO[] operationDOs = pm.getOperations(serviceDO);
                    if (operationDOs != null) {
                        for (int j = 0; j < operationDOs.length; j++) {
                            OperationDO operationDO = operationDOs[j];
                            OperationParameterDO[] operationParameterDOs =
                                    pm.getOperationParameters(operationDO);
                            OperationParameterDO operationParameterDO = null;
                            for (int k = 0; k < operationParameterDOs.length; k++) {
                                if (operationParameterDOs[k].getName().equals(parameter.getName())) {
                                    operationParameterDO = operationParameterDOs[k];
                                    break;
                                }
                            }
                            if (operationParameterDO != null) {
                                operationParameterDO
                                        .setValue(parameter.getParameterElement().toString());
                                pm.updateEntity(operationParameterDO);
                            } else {
                                operationParameterDO = new OperationParameterDO();
                                operationParameterDO.setName(parameter.getName());
                                operationParameterDO
                                        .setValue(parameter.getParameterElement().toString());
                                operationParameterDO.setOperation(operationDO);
                                try {
                                    pm.addEntity(operationParameterDO);
                                } catch (DuplicateEntityException e) {
                                    log.error("Service Parameter already exists", e);
                                }

                            }

                        }
                    }
                }
            }

        }
        return listServiceGroup(serviceGroupName);

    }

    /**
     * List all the available service groups
     */
    public ServiceGroupMetaData[] listServiceGroups() throws AxisFault {

        ArrayList sgList = new ArrayList();
        boolean clientsideService;

        for (Iterator sgs = getAxisConfig().getServiceGroups(); sgs.hasNext();) {
            clientsideService = false;
            AxisServiceGroup axisServiceGroup = (AxisServiceGroup) sgs.next();

            String sgName = axisServiceGroup.getServiceGroupName();
            if (SystemFilter.isFilteredOutService(sgName)) {
                // No need to display Admin Service Groups
                continue;
            }

            ServiceGroupMetaData sgMetaData = new ServiceGroupMetaData();
            sgMetaData.setServiceContextPath(getConfigContext().getServiceContextPath());
            List services = new ArrayList();
            List serviceTypes = new ArrayList();
            Parameter serviceTypeParam;
            for (Iterator serviceIter = axisServiceGroup.getServices(); serviceIter.hasNext();) {
                AxisService axisService = (AxisService) serviceIter.next();
                if (axisService.isClientSide()) {
                    clientsideService = true;
                }
                services.add(axisService.getName());
                //extract service type
                serviceTypeParam = axisService.getParameter(ServerConstants.SERVICE_TYPE);
                if (serviceTypeParam != null) {
                    serviceTypes.add(
                            axisService.getName() + "#" + (String) serviceTypeParam.getValue());
                } else {
                    serviceTypes
                            .add(axisService.getName() + "#" + ServerConstants.SERVICE_TYPE_OTHER);
                }
            }

            sgMetaData.setServices((String[]) services.toArray(new String[services.size()]));
            sgMetaData.setServiceTypes(
                    (String[]) serviceTypes.toArray(new String[serviceTypes.size()]));
            sgMetaData.setGroup_id(sgName);
            sgMetaData.setGroup_name(sgName);
            sgMetaData.setGroup_version("-");

            if (!clientsideService) {
                sgList.add(sgMetaData);
            }

            Parameter parameter =
                    axisServiceGroup.getParameter(Constants.Configuration.ENABLE_MTOM);
            if (parameter != null) {
                sgMetaData.setEnableMTOM((String) parameter.getValue());
            } else {
                sgMetaData.setEnableMTOM("false");
            }


        }
        Collections.sort(sgList, new Comparator() {
            public int compare(Object arg0, Object arg1) {
                ServiceGroupMetaData a = (ServiceGroupMetaData) arg0;
                ServiceGroupMetaData b = (ServiceGroupMetaData) arg1;
                return a.getGroup_name().compareToIgnoreCase(b.getGroup_name());
            }
        });
        return (ServiceGroupMetaData[]) sgList.toArray(new ServiceGroupMetaData[sgList.size()]);
    }

    /**
     * Remove a service group
     *
     * @param groupName
     * @throws AxisFault
     */
    public void removeServiceGroup(String groupName) throws AxisFault {
        AxisServiceGroup asg = getAxisConfig().getServiceGroup(groupName);
        if (asg == null) {
            throw new AxisFault("Invalid service group name: " + groupName);
        }
        getAxisConfig().removeServiceGroup(groupName);
    }

    /**
     * Return all parameters for this service group (including inherited ones), where each parameter
     * is an XML fragment representing the "parameter" element
     *
     * @param service_group_id
     * @param group_version
     * @return list of service group params as on OMElement[]
     * @throws AxisFault
     */
    public OMElement[] getServiceGroupParameters(String service_group_id,
                                                 String group_version) throws AxisFault {
        List allParameter = new ArrayList();
        List globalParameters = getAxisConfig().getParameters();

        for (int i = 0; i < globalParameters.size(); i++) {
            Parameter parameter = (Parameter) globalParameters.get(i);
            allParameter.add(parameter.getParameterElement());
        }
        AxisServiceGroup asg = getAxisConfig().getServiceGroup(service_group_id);
        List sgParameters = asg.getParameters();
        for (int i = 0; i < sgParameters.size(); i++) {
            Parameter parameter = (Parameter) sgParameters.get(i);
            allParameter.add(parameter.getParameterElement());
        }
        return (OMElement[]) allParameter.toArray(new OMElement[allParameter.size()]);
    }

    /**
     * return only the parameters for explicitly set for this service group (not including inherited
     * ones), where each parameter is an XML fragment representing the "parameter" element
     *
     * @param serviceGroupId
     * @return list of declared service group params as an OMElement[]
     * @throws AxisFault
     */
    public OMElement[] getDeclaredServiceGroupParameters(String serviceGroupId) throws AxisFault {
        List allParameter = new ArrayList();
        AxisServiceGroup asg = getAxisConfig().getServiceGroup(serviceGroupId);
        List sgParameters = asg.getParameters();

        for (int i = 0; i < sgParameters.size(); i++) {
            Parameter parameter = (Parameter) sgParameters.get(i);
            OMElement element = parameter.getParameterElement();
            if (element != null) {
                allParameter.add(element);
            }
        }

        return (OMElement[]) allParameter.toArray(new OMElement[allParameter.size()]);
    }

    public void setServiceGroupParameters(String serviceGroupId,
                                          OMElement parameterElement[]) throws AxisFault {
        for (int i = 0; i < parameterElement.length; i++) {
            setServiceGroupParameter(serviceGroupId, parameterElement[i]);
        }
    }

    public void setServiceGroupParameter(String serviceGroupId,
                                         OMElement parameterElement) throws AxisFault {

        AxisServiceGroup axisServiceGroup = getAxisConfig().getServiceGroup(serviceGroupId);
        Parameter parameter = ParameterUtil.createParameter(parameterElement);
        Parameter p = axisServiceGroup.getParameter(parameter.getName());
        ServiceGroupDO serviceGroupDO = pm.getServiceGroup(serviceGroupId);
        ServiceGroupParameterDO sgParamDO = serviceGroupDO.getParameter(parameter.getName());
        if (p != null && sgParamDO != null) {

            // Setting a new value for a parameter declared in AxisServiceGroup
            if (!p.isLocked()) {
                axisServiceGroup.addParameter(parameter);
                sgParamDO.setValue(parameterElement.toString());
                pm.updateEntity(sgParamDO);
            }
        } else {

            // If we are adding a new AxisServiceGroup param or overriding a param in the Configuration hierarchy
            if (p == null || !p.isLocked()) {
                axisServiceGroup.addParameter(parameter);

                sgParamDO = new ServiceGroupParameterDO();
                sgParamDO.setName(parameter.getName());
                sgParamDO.setValue(parameterElement.toString());
                sgParamDO.setServiceGroup(serviceGroupDO);

                try {
                    pm.addEntity(sgParamDO);
                } catch (DuplicateEntityException e) {
                    log.error("Service Parameter already exists", e);
                }
            }
        }
    }

    public void removeServiceGroupParameter(String serviceGroupId,
                                            OMElement parameterElement) throws AxisFault {

        AxisServiceGroup axisServiceGroup = getAxisConfig().getServiceGroup(serviceGroupId);
        Parameter parameter = ParameterUtil.createParameter(parameterElement);
        Parameter p = axisServiceGroup.getParameter(parameter.getName());
        ServiceGroupDO serviceGroupDO = pm.getServiceGroup(serviceGroupId);
        ServiceGroupParameterDO sgParamDO = serviceGroupDO.getParameter(parameter.getName());
        if (p != null && sgParamDO != null) {
            if (!p.isLocked()) {
                axisServiceGroup.removeParameter(parameter);
                pm.deleteEntity(sgParamDO);
            }
        } else {

            if (p == null || !p.isLocked()) {
                axisServiceGroup.removeParameter(parameter);
            }
        }
    }

    public boolean deleteServiceGroup(String serviceGroupName) throws AxisFault {
        // We cannot delete items from a URL repo
        String axis2Repo = ServerConfiguration.getInstance().
                getFirstProperty(ServerConfiguration.AXIS2_CONFIG_REPO_LOCATION);
        if (WsasUtils.isURL(axis2Repo)) {
            throw new AxisFault("You are not permitted to remove the " +
                                serviceGroupName +
                                " service group from the URL repository " + axis2Repo);
        }

        AxisConfiguration axisConfig = getAxisConfig();
        AxisServiceGroup asGroup = axisConfig.getServiceGroup(serviceGroupName);

        if (asGroup == null) {
            throw new AxisFault("Invalid service group name " + serviceGroupName);
        }

        String fileName = "";
        Parameter hotDeployment = axisConfig.getParameter("hotdeployment");
        boolean hotdeploymentEnabled = true;
        if (hotDeployment != null) {
            hotdeploymentEnabled =
                    Boolean.valueOf(((String) hotDeployment.getValue())).booleanValue();
        }
        List serviceToBeRemoved = new ArrayList();
        for (Iterator serviceIter = asGroup.getServices(); serviceIter.hasNext();) {
            AxisService axisService = (AxisService) serviceIter.next();
            URL fn = axisService.getFileName();
            if (fn != null) {
                fileName = fn.getPath();
            }
            if ((fileName != null) && (fileName.trim().length() != 0)) {
                File file = new File(fileName);
                if (file.exists()) {
                    if (!((file.isDirectory() && new FileManipulator().deleteDir(file)) ||
                          file.delete())) {
                        throw new AxisFault("Service archive deletion failed. " +
                                            "Due to a JVM issue on MS-Windows, " +
                                            "AAR files cannot be deleted!");
                    }
                } else {
                    throw new AxisFault("Service archive deletion failed since the file " +
                                        fileName + " cannot be found");
                }
            } else {
                 throw new AxisFault("Service archive deletion failed since the " +
                                     "filename related to the services is not set");
            }
            if (!hotdeploymentEnabled) {
                serviceToBeRemoved.add(axisService.getName());
            }
            //NOTE: The DeploymentInterceptor will handle removing the service from the DB
        }
        if (!hotdeploymentEnabled) {
            for (Iterator iterator = serviceToBeRemoved.iterator(); iterator.hasNext();) {
                axisConfig.removeService((String) iterator.next());
            }
            axisConfig.removeServiceGroup(asGroup.getServiceGroupName());
        }
        return true;
    }

    public OMElement getPolicy(String serviceGroupId,
                               String serviceGroupVersion) throws AxisFault {
        AxisServiceGroup axisServiceGroup = getAxisConfig().getServiceGroup(serviceGroupId);

        if (axisServiceGroup == null) {
            throw new AxisFault("Invalid AxisServiceGroup: " + serviceGroupId);
        }

        PolicyInclude groupPolicyInclude = axisServiceGroup.getPolicyInclude();
        Policy groupPolicy = groupPolicyInclude.getPolicy();

        if (groupPolicy == null) {
            return PolicyUtil.getEmptyPolicyAsOMElement();
        }

        return PolicyUtil.getPolicyAsOMElement(groupPolicy);
    }

    public void setPolicy(String serviceGroupId, String serviceGroupVersion,
                          OMElement policyElement) throws AxisFault {
        AxisServiceGroup axisServiceGroup = getAxisConfig().getServiceGroup(serviceGroupId);

        if (axisServiceGroup == null) {
            throw new AxisFault("Invalid AxisServiceGroup: " + serviceGroupId);
        }

        Policy groupPolicy;

        try {
            groupPolicy = PolicyUtil.getPolicyFromOMElement(policyElement);
        } catch (RuntimeException ex) {
            throw new AxisFault("cannot convert the OMElement to Policy", ex);
        }

        PolicyInclude groupPolicyInclude = axisServiceGroup.getPolicyInclude();
        groupPolicyInclude.setPolicy(groupPolicy);
    }

    /**
     * Using the information from AxisServiceGroup, a service archive will be created. A String will
     * be returned with ID, that can be used to access the AAR and dump it anywhere user wishes.
     *
     * @param serviceGroupName
     * @return id of service archive
     * @throws AxisFault
     */
    public String dumpAAR(String serviceGroupName) throws AxisFault {
        return ServiceArchiveCreator.createArchive(getConfigContext(), serviceGroupName);

    }
}
