/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.admin.service;

import org.apache.axiom.om.OMAbstractFactory;
import org.apache.axiom.om.OMAttribute;
import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.OMFactory;
import org.apache.axiom.om.OMNamespace;
import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.apache.axis2.AxisFault;
import org.apache.axis2.Constants;
import org.apache.axis2.context.ConfigurationContext;
import org.apache.axis2.context.MessageContext;
import org.apache.axis2.description.AxisOperation;
import org.apache.axis2.description.AxisService;
import org.apache.axis2.description.AxisServiceGroup;
import org.apache.axis2.description.Parameter;
import org.apache.axis2.description.PolicyInclude;
import org.apache.axis2.description.TransportInDescription;
import org.apache.axis2.engine.AxisConfiguration;
import org.apache.axis2.rpc.receivers.RPCInOnlyMessageReceiver;
import org.apache.axis2.rpc.receivers.RPCMessageReceiver;
import org.apache.axis2.rpc.receivers.ejb.EJBUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.neethi.Policy;
import org.apache.neethi.PolicyRegistry;
import org.springframework.context.ApplicationContext;
import org.wso2.utils.AbstractAdmin;
import org.wso2.utils.ArchiveManipulator;
import org.wso2.utils.FileManipulator;
import org.wso2.utils.MBeanRegistrar;
import org.wso2.wsas.ServerConstants;
import org.wso2.wsas.admin.service.spring.GenericApplicationContextSupplier;
import org.wso2.wsas.admin.service.util.ClassMethodsData;
import org.wso2.wsas.admin.service.util.FaultService;
import org.wso2.wsas.admin.service.util.GenericApplicationContextUtil;
import org.wso2.wsas.admin.service.util.MethodData;
import org.wso2.wsas.admin.service.util.PolicyData;
import org.wso2.wsas.admin.service.util.ServiceMetaData;
import org.wso2.wsas.admin.service.util.ServiceRoleData;
import org.wso2.wsas.admin.service.util.ServiceUserData;
import org.wso2.wsas.admin.service.util.SpringBeansData;
import org.wso2.wsas.persistence.PersistenceManager;
import org.wso2.wsas.persistence.dataobject.EJBAppServerDO;
import org.wso2.wsas.persistence.dataobject.EJBProviderDO;
import org.wso2.wsas.persistence.dataobject.OperationDO;
import org.wso2.wsas.persistence.dataobject.OperationParameterDO;
import org.wso2.wsas.persistence.dataobject.ServiceDO;
import org.wso2.wsas.persistence.dataobject.ServiceIdentifierDO;
import org.wso2.wsas.persistence.dataobject.ServiceParameterDO;
import org.wso2.wsas.persistence.dataobject.ServicePolicyDO;
import org.wso2.wsas.persistence.dataobject.ServiceUserDO;
import org.wso2.wsas.persistence.dataobject.ServiceUserRoleDO;
import org.wso2.wsas.persistence.dataobject.TransportDO;
import org.wso2.wsas.persistence.exception.DuplicateEntityException;
import org.wso2.wsas.persistence.exception.ServiceNotFoundException;
import org.wso2.wsas.persistence.exception.ServicePolicyAlreadyExistsException;
import org.wso2.wsas.persistence.exception.ServicePolicyNotFoundException;
import org.wso2.wsas.transport.util.TransportSummary;
import org.wso2.wsas.util.ParameterUtil;
import org.wso2.wsas.util.PolicyUtil;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Admin service to manage services
 */
public class ServiceAdmin extends AbstractAdmin implements ServiceAdminMBean {
    private static final Log log = LogFactory.getLog(ServiceAdmin.class);

    private static final String SERVICE_MUST_CONTAIN_AT_LEAST_ONE_TRANSPORT =
            "Cannot remove transport binding. " +
            "<br/>A service must contain at least one transport binding!";
    private PersistenceManager pm = new PersistenceManager();

    static {
        MBeanRegistrar.registerMBean(new ServiceAdmin());
    }

    public ServiceAdmin() {
    }

    /**
     * Expose this service only via the specified transport
     *
     * @param serviceId
     * @param transportProtocols
     * @return Status Message
     * @throws AxisFault
     */
    public String exposeServiceOnlyViaTransports(String serviceId,
                                                 String[] transportProtocols) throws AxisFault {

        AxisService axisService = getAxisConfig().getService(serviceId);
        ArrayList transports = new ArrayList();
        for (int i = 0; i < transportProtocols.length; i++) {
            transports.add(transportProtocols[i]);
        }
        axisService.setExposedTransports(transports);

        ServiceDO serviceDO = pm.getService(axisService.getName(),
                                            ServiceIdentifierDO.EMPTY_SERVICE_VERSION);
        serviceDO.setIsExposedOnAllTransports(false);

        TransportDO[] transportDOs = pm.getTransports();
        for (int i = 0; i < transportDOs.length; i++) { // Remove all transports
            serviceDO.getTransports().remove(transportDOs[i]);
        }
        for (int i = 0; i < transportProtocols.length; i++) {
            TransportDO transportDO = pm.getTransport(transportProtocols[i]);
            serviceDO.addTransport(transportDO); // add only specified transports
            pm.updateEntity(transportDO);
        }

        try {
            pm.updateService(serviceDO);
        } catch (ServiceNotFoundException e) {
            String msg = "Service with name " + serviceId + " not found.";
            log.error(msg);
            throw new AxisFault(msg, e);
        }
        return "Successfully add selected transport bindings to service " + serviceId;
    }

    public String addTransportBinding(String serviceId,
                                      String transportProtocol) throws AxisFault {
        if (transportProtocol == null || transportProtocol.trim().length() == 0) {
            return "Invalid transport " + transportProtocol;
        }
        AxisService axisService = getAxisConfig().getService(serviceId);

        if (axisService.isExposedTransport(transportProtocol)) {
            return "Service [" + serviceId + "] already contains the " + transportProtocol +
                   " transport binding!";
        }

        ServiceDO serviceDO = pm.getService(axisService.getName(),
                                            ServiceIdentifierDO.EMPTY_SERVICE_VERSION);
        if (serviceDO.getIsUTAuthEnabled()) {
            if (!transportProtocol.equalsIgnoreCase(ServerConstants.HTTPS_TRANSPORT)) {
                throw new AxisFault("Cannot add non-HTTPS transport binding for Service [" +
                                    serviceId + "] since a security scenario which requires the " +
                                    "service to contain only the HTTPS transport binding" +
                                    " has been applied to this service.");
            }
        }

        if (!axisService.isEnableAllTransports()) {
            axisService.addExposedTransport(transportProtocol);
        } else {
            return "Service [" + serviceId + "] already contains the " + transportProtocol +
                   " transport binding!";
        }

        TransportDO transportDO = pm.getTransport(transportProtocol);
        if (transportDO != null) {
            serviceDO.setIsExposedOnAllTransports(false);
            serviceDO.addTransport(transportDO);

            try {
                pm.updateService(serviceDO);
                pm.updateEntity(transportDO);
            } catch (ServiceNotFoundException e) {
                String msg = "Service with name " + serviceId + " not found.";
                log.error(msg);
                throw new AxisFault(msg, e);
            }
        }

        return "Successfully added " + transportProtocol + " transport binding to service " +
               serviceId;
    }

    public String removeTransportBinding(String serviceId,
                                         String transportProtocol) throws AxisFault {

        AxisService axisService = getAxisConfig().getService(serviceId);
        ServiceDO serviceDO = pm.getService(axisService.getName(),
                                            ServiceIdentifierDO.EMPTY_SERVICE_VERSION);
        if (serviceDO.getIsUTAuthEnabled()) {
            if (transportProtocol.equalsIgnoreCase(ServerConstants.HTTPS_TRANSPORT)) {
                throw new AxisFault("HTTPS transport binding for Service [" + serviceId +
                                    "] cannot be removed since a security scenario which requires" +
                                    " HTTPS has been applied to this service.");
            }
        }
        TransportDO transportDO = pm.getTransport(transportProtocol);
        if (!axisService.isEnableAllTransports()) {
            if (axisService.getExposedTransports().size() == 1) {
                return SERVICE_MUST_CONTAIN_AT_LEAST_ONE_TRANSPORT;
            } else {
                serviceDO.getTransports().remove(transportDO);
                transportDO.getServices().remove(serviceDO);
                axisService.removeExposedTransport(transportProtocol);
                pm.updateEntity(transportDO);
            }
        } else {
            TransportSummary[] transports = new TransportAdmin().listTransports();

            if (transports.length == 1) {
                return SERVICE_MUST_CONTAIN_AT_LEAST_ONE_TRANSPORT;
            } else {
                for (int i = 0; i < transports.length; i++) {
                    String protocol = transports[i].getProtocol();

                    if (!protocol.equals(transportProtocol)) {
                        axisService.addExposedTransport(protocol);
                        TransportDO addTransportDO = pm.getTransport(protocol);
                        serviceDO.addTransport(addTransportDO);
                        pm.updateEntity(addTransportDO);
                    }
                }
            }
        }

        serviceDO.setIsExposedOnAllTransports(false);
        try {
            pm.updateService(serviceDO);
        } catch (ServiceNotFoundException e) {
            String msg = "Service with name " + serviceId + " not found.";
            log.error(msg);
            throw new AxisFault(msg, e);
        }

        return "Removed " + transportProtocol + " transport binding for " + serviceId + " service";
    }

    public String[] getExposedTransports(String serviceId) throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceId);
        if (!axisService.isEnableAllTransports()) {
            List exposedTransports = axisService.getExposedTransports();
            return (String[]) exposedTransports.toArray(new String[exposedTransports.size()]);
        } else {
            HashMap transportsIn = getConfigContext().getAxisConfiguration()
                    .getTransportsIn();
            String[] transports = new String[transportsIn.size()];
            int i = 0;

            for (Iterator transportsIter = transportsIn.values().iterator();
                 transportsIter.hasNext();) {
                TransportInDescription tiDesc = (TransportInDescription) transportsIter.next();
                transports[i++] = tiDesc.getName();
            }

            return transports;
        }
    }

    public void creatService(String serviceName) throws AxisFault {
        AxisService service = AxisService.createService(serviceName,
                                                        getAxisConfig());
        getAxisConfig().addService(service);
    }

    public void stopService(String serviceName) throws Exception {
        AxisService axisService = getAxisConfig().getServiceForActivation(serviceName);
        if (axisService == null) {
            throw new Exception("Invalid service name " + serviceName);
        }

        try {
            getAxisConfig().stopService(serviceName);
        } catch (AxisFault e) {
            String msg = "Cannot stop service " + serviceName;
            log.error(msg, e);
            throw new Exception(msg);
        }
    }

    /* public String[] getActiveEndPoints() {
List endpoints = new ArrayList();
for (Iterator iter = getAxisConfig().getServices().values().iterator(); iter.hasNext();) {
AxisService service = (AxisService) iter.next();
if (service.isActive()) {
    *//* String[] eprs;
                try {
                    eprs = service.getEPRs();
                } catch (AxisFault e) {
                    throw new RuntimeException("Cannot get service EPRs", e);
                }
                for (int i = 0; i < eprs.length; i++) {
                    String epr = eprs[i];
                    if (!endpoints.contains(epr)) {
                        endpoints.add(epr);
                    }
                }*//*

                for (Iterator endpointIter = service.getEndpoints().values().iterator();
                     endpointIter.hasNext();) {
                    AxisEndpoint axisEndpoint = (AxisEndpoint) endpointIter.next();
                    String binding = axisEndpoint.getBinding().getName().toString();
                    String epr = axisEndpoint.getEndpointURL();

                    endpoints.add(epr + " [" + binding + "]");

                }
            }
        }
        return (String[]) endpoints.toArray(new String[endpoints.size()]);
    }
*/

    public int getNumberOfActiveServices() throws Exception {
        int activeServices = 0;
        HashMap services = getAxisConfig().getServices();
        for (Iterator iterator = services.values().iterator(); iterator.hasNext();) {
            AxisService service = (AxisService) iterator.next();
            if (service.isActive()) {
                activeServices++;
            }
        }
        return activeServices;
    }

    public int getNumberOfInactiveServices() throws Exception {
        int inactiveServices = 0;
        HashMap services = getAxisConfig().getServices();
        for (Iterator iterator = services.values().iterator(); iterator.hasNext();) {
            AxisService service = (AxisService) iterator.next();
            if (!service.isActive()) {
                inactiveServices++;
            }
        }
        return inactiveServices;
    }

    public void startService(String serviceName) throws Exception {
        AxisService axisService = getAxisConfig().getServiceForActivation(serviceName);

        if (axisService == null) {
            throw new Exception("Invalid service name " + serviceName);
        }
        try {
            getAxisConfig().startService(serviceName);
        } catch (AxisFault e) {
            String msg = "Cannot start service " + serviceName;
            log.error(msg, e);
            throw new Exception(msg);
        }
    }

    private String[] getServiceEPRs(AxisService service) throws AxisFault {
        if (!service.isActive()) {
            return new String[]{"Service is inactive. No valid end point references."};
        }
        return service.getEPRs();
    }

    public ServiceMetaData getServiceData(String serviceName) throws AxisFault {
        AxisService service = getAxisConfig().getServiceForActivation(serviceName);
        Parameter serviceTypeParam = service.getParameter(ServerConstants.SERVICE_TYPE);
        String serviceType;
        if (serviceTypeParam != null) {
            serviceType = (String) serviceTypeParam.getValue();
        } else {
            serviceType = "Normal";
        }
        List ops = new ArrayList();
        for (Iterator opIter = service.getOperations(); opIter.hasNext();) {
            AxisOperation axisOperation = (AxisOperation) opIter.next();

            if (axisOperation.getName() != null) {
                ops.add(axisOperation.getName().getLocalPart());
            }
        }

        ServiceMetaData smd = new ServiceMetaData();

        try {
            StatisticsClient statClient = new StatisticsClient();
            smd.setServiceRequestCount(statClient.getServiceRequestCount(serviceName));
            smd.setServiceResponseCount(statClient.getServiceResponseCount(serviceName));
            smd.setServiceFaultCount(statClient.getServiceFaultCount(serviceName));
            smd.setMaxResponseTime(statClient.getMaxServiceResponseTime(serviceName));
            smd.setMinResponseTime(statClient.getMinServiceResponseTime(serviceName));
            smd.setAvgResponseTime(statClient.getAvgServiceResponseTime(serviceName));
        } catch (Exception e) {
            String msg = "Could not get service statistics";
            log.error(msg, e);
            throw new AxisFault(msg, e);
        }

        smd.setOperations((String[]) ops.toArray(new String[ops.size()]));
        smd.setName(serviceName);
        smd.setServiceId(serviceName);
        smd.setServiceVersion("");
        smd.setActive(service.isActive());
        smd.setEprs(getServiceEPRs(service));
        smd.setServiceType(serviceType);
        AxisServiceGroup serviceGroup = (AxisServiceGroup) service.getParent();
        smd.setFoundWebResources(serviceGroup.isFoundWebResources());
        smd.setScope(service.getScope());
        smd.setServiceGroupName(serviceGroup.getServiceGroupName());

        if (service.getDocumentation() != null) {
            smd.setDescription(service.getDocumentation());
        } else {
            smd.setDescription("No service description found");
        }

        Parameter parameter = service.getParameter(Constants.Configuration.ENABLE_MTOM);
        if (parameter != null) {
            smd.setEnableMTOM((String) parameter.getValue());
        } else {
            smd.setEnableMTOM("false");
        }

        return smd;
    }

    /**
     * set the service parameter enableMTOM to manipulate MTOM flag true/false/optional
     *
     * @param flag
     * @param serviceName
     */
    public void configureMTOM(String flag, String serviceName) throws AxisFault {

        AxisService service = getAxisConfig().getService(serviceName);
        if (service == null) {
            throw new AxisFault("AxisService " + serviceName + " cannot be found.");
        }

        ArrayList parameters = service.getParameters();

        boolean found = false;
        for (Iterator iterator = parameters.iterator(); iterator.hasNext();) {
            Parameter parameter = (Parameter) iterator.next();

            if (parameter.getParameterType() == Parameter.TEXT_PARAMETER &&
                parameter.getValue().toString().equals(Constants.Configuration.ENABLE_MTOM)) {
                parameter.setValue(flag.trim());
                found = true;
                break;
            }

        }
        if (!found) {
            Parameter parameter = ParameterUtil
                    .createParameter(Constants.Configuration.ENABLE_MTOM, flag.trim());
            service.addParameter(parameter);
        }

        Parameter parameter = service.getParameter(Constants.Configuration.ENABLE_MTOM);
        for (Iterator iterator1 = service.getOperations(); iterator1.hasNext();) {
            AxisOperation axisOperation = (AxisOperation) iterator1.next();
            axisOperation.addParameter(ParameterUtil.createParameter(
                    Constants.Configuration.ENABLE_MTOM, (String) parameter.getValue()));
        }
        ServiceDO serviceDO = pm.getService(serviceName,
                                            ServiceIdentifierDO.EMPTY_SERVICE_VERSION);
        ServiceParameterDO paramDO = pm.getServiceParameter(serviceDO,
                                                            parameter.getName());
        if (paramDO != null) {
            paramDO.setValue(parameter.getParameterElement().toString());
            pm.updateEntity(paramDO);
        } else {
            ServiceParameterDO param = new ServiceParameterDO();
            param.setName(parameter.getName());
            param.setValue(parameter.getParameterElement().toString());
            param.setService(serviceDO);

            try {
                pm.addEntity(param);
            } catch (DuplicateEntityException e) {
                log.error("Service Parameter already exists", e);
            }

        }


        OperationDO[] operationDOs = pm.getOperations(serviceDO);
        if (operationDOs != null) {
            for (int j = 0; j < operationDOs.length; j++) {
                OperationDO operationDO = operationDOs[j];
                OperationParameterDO[] operationParameterDOs =
                        pm.getOperationParameters(operationDO);
                OperationParameterDO operationParameterDO = null;
                for (int k = 0; k < operationParameterDOs.length; k++) {
                    if (operationParameterDOs[k].getName().equals(parameter.getName())) {
                        operationParameterDO = operationParameterDOs[k];
                        break;
                    }
                }
                if (operationParameterDO != null) {
                    operationParameterDO
                            .setValue(parameter.getParameterElement().toString());
                    pm.updateEntity(operationParameterDO);
                } else {
                    operationParameterDO = new OperationParameterDO();
                    operationParameterDO.setName(parameter.getName());
                    operationParameterDO
                            .setValue(parameter.getParameterElement().toString());
                    operationParameterDO.setOperation(operationDO);
                    try {
                        pm.addEntity(operationParameterDO);
                    } catch (DuplicateEntityException e) {
                        log.error("Service Parameter already exists", e);
                    }

                }

            }
        }
    }

    /**
     * @param serviceGroupId
     * @return list of services
     * @throws AxisFault
     */
    public ServiceMetaData[] listServices(String serviceGroupId) throws AxisFault {
        AxisServiceGroup asg = getAxisConfig().getServiceGroup(serviceGroupId);
        List services = new ArrayList();

        if (asg == null) {
            throw new AxisFault("Invalid service group name");
        } else {
            for (Iterator serviceIter = asg.getServices(); serviceIter.hasNext();) {
                AxisService axisService = (AxisService) serviceIter.next();
                Parameter serviceTypeParam = axisService.getParameter(ServerConstants.SERVICE_TYPE);
                String serviceType;
                if (serviceTypeParam != null) {
                    serviceType = (String) serviceTypeParam.getValue();
                } else {
                    serviceType = "Normal";
                }
                List ops = new ArrayList();
                for (Iterator opIter = axisService.getOperations(); opIter.hasNext();) {
                    AxisOperation axisOperation = (AxisOperation) opIter.next();
                    ops.add(axisOperation.getName().getLocalPart());
                }

                ServiceMetaData smd = new ServiceMetaData();
                String serviceName = axisService.getName();
                try {
                    StatisticsClient statClient = new StatisticsClient();
                    smd.setServiceRequestCount(statClient.getServiceRequestCount(serviceName));
                    smd.setServiceResponseCount(statClient.getServiceResponseCount(serviceName));
                    smd.setServiceFaultCount(statClient.getServiceFaultCount(serviceName));
                    smd.setMaxResponseTime(statClient.getMaxServiceResponseTime(serviceName));
                    smd.setMinResponseTime(statClient.getMinServiceResponseTime(serviceName));
                    smd.setAvgResponseTime(statClient.getAvgServiceResponseTime(serviceName));
                } catch (Exception e) {
                    throw AxisFault.makeFault(e);
                }
                smd.setOperations((String[]) ops.toArray(new String[ops.size()]));
                smd.setName(serviceName);
                smd.setServiceId(serviceName);
                smd.setEprs(getServiceEPRs(axisService));
                smd.setServiceVersion("");
                smd.setActive(axisService.isActive());
                smd.setServiceType(serviceType);
                smd.setDescription(axisService.getDocumentation());
                services.add(smd);
            }
        }
        Collections.sort(services, new Comparator() {
            public int compare(Object arg0, Object arg1) {
                ServiceMetaData a = (ServiceMetaData) arg0;
                ServiceMetaData b = (ServiceMetaData) arg1;
                return a.getName().compareToIgnoreCase(b.getName());
            }
        });

        return (ServiceMetaData[]) services.toArray(new ServiceMetaData[services.size()]);
    }

    // return all available service metadata (not counts)
    public Object[] getServiceInfo(String serviceId, String serviceVersion) throws AxisFault {
        throw new AxisFault("Operation not yet implemented");
    }

    // return all accumulated data about this service
    public Object[] getServiceStatus(String serviceId, String serviceVersion) throws AxisFault {
        throw new AxisFault("Operation not yet implemented");
    }

    // return all parameters for this service (including inherited ones),
    // where each parameter is an XML fragment representing the "parameter"
    // element
    public OMElement[] getServiceParameters(String serviceId,
                                            String serviceVersion) throws AxisFault {
        ArrayList allParameters = new ArrayList();
        ArrayList globalParameters = getAxisConfig().getParameters();

        for (int i = 0; i < globalParameters.size(); i++) {
            Parameter parameter = (Parameter) globalParameters.get(i);
            allParameters.add(parameter.getParameterElement());
        }

        AxisService service = getAxisConfig().getService(serviceId);

        if (service == null) {
            throw new AxisFault("Invalid service name");
        }

        ArrayList serviceParams = service.getParameters();

        for (int i = 0; i < serviceParams.size(); i++) {
            Parameter parameter = (Parameter) serviceParams.get(i);

            String paramName = parameter.getName();
            if (paramName != null && paramName.trim().length() != 0) {
                if (parameter.getParameterElement() != null) {
                    allParameters.add(parameter.getParameterElement());
                } else if (parameter.getValue() != null && parameter.getValue() instanceof String) {
                    parameter = ParameterUtil.createParameter(paramName.trim(),
                                                              (String) parameter.getValue());
                    allParameters.add(parameter.getParameterElement());
                }
            }
        }
        return (OMElement[]) allParameters.toArray(new OMElement[allParameters.size()]);
    }

    // return only the parameters for explicitly set for this service (not
    // including inherited ones),
    // where each parameter is an XML fragment representing the <parameter>
    // element
    public OMElement[] getDeclaredServiceParameters(String serviceId,
                                                    String serviceVersion) throws AxisFault {
        ArrayList allParameters = new ArrayList();
        AxisService axisService = getAxisConfig().getService(serviceId);

        if (axisService == null) {
            throw new AxisFault("invalid service name " + serviceId);
        }

        ArrayList params = axisService.getParameters();

        for (int i = 0; i < params.size(); i++) {
            Parameter parameter = (Parameter) params.get(i);
            String paramName = parameter.getName();
            if (paramName != null && paramName.trim().length() != 0) {
                if (parameter.getParameterElement() != null) {
                    allParameters.add(parameter.getParameterElement());
                } else if (parameter.getValue() != null && parameter.getValue() instanceof String) {
                    parameter = ParameterUtil.createParameter(paramName.trim(),
                                                              (String) parameter.getValue());
                    allParameters.add(parameter.getParameterElement());
                }
            }
        }

        return (OMElement[]) allParameters.toArray(new OMElement[allParameters.size()]);
    }

    public void setServiceParameters(String serviceId,
                                     String serviceVersion,
                                     OMElement[] paramterElements) throws AxisFault {
        for (int i = 0; i < paramterElements.length; i++) {
            setServiceParameter(serviceId, serviceVersion, paramterElements[i]);
        }

    }

    public void setServiceParameter3(String serviceId,
                                     Parameter parameter) throws AxisFault {

        AxisService axisService = getAxisConfig().getService(serviceId);

        if (axisService == null) {
            throw new AxisFault("Invalid service name '" + serviceId + "'");
        }

        // Persist the parameter
        ServiceDO serviceDO = pm.getService(serviceId,
                                            ServiceIdentifierDO.EMPTY_SERVICE_VERSION);
        Parameter p = axisService.getParameter(parameter.getName());
        ServiceParameterDO paramDO = pm.getServiceParameter(serviceDO,
                                                            parameter.getName());
        if (p != null && paramDO != null) {

            // Setting a new value for a parameter declared in AxisService
            if (!p.isLocked()) {
                axisService.addParameter(parameter);
                paramDO.setValue(parameter.getParameterElement().toString());
                pm.updateEntity(paramDO);
            }
        } else {

            // If we are adding a new AxisService param or overriding a param in the Configuration hierarchy
            if (p == null || !p.isLocked()) {
                axisService.addParameter(parameter);

                ServiceParameterDO param = new ServiceParameterDO();
                param.setName(parameter.getName());
                param.setValue(parameter.getParameterElement().toString());
                param.setService(serviceDO);

                try {
                    pm.addEntity(param);
                } catch (DuplicateEntityException e) {
                    log.error("Service Parameter already exists", e);
                }
            }
        }
    }

    public void setServiceParameter(String serviceId,
                                    String serviceVersion,
                                    OMElement paramterElement) throws AxisFault {

        AxisService axisService = getAxisConfig().getService(serviceId);

        if (axisService == null) {
            throw new AxisFault("Invalid service name '" + serviceId + "'");
        }

        // Persist the parameter
        ServiceDO serviceDO = pm.getService(serviceId,
                                            ServiceIdentifierDO.EMPTY_SERVICE_VERSION);
        Parameter parameter = ParameterUtil.createParameter(paramterElement);
        Parameter p = axisService.getParameter(parameter.getName());
        ServiceParameterDO paramDO = pm.getServiceParameter(serviceDO,
                                                            parameter.getName());
        if (p != null && paramDO != null) {

            // Setting a new value for a parameter declared in AxisService
            if (!p.isLocked()) {
                axisService.addParameter(parameter);
                paramDO.setValue(paramterElement.toString());
                pm.updateEntity(paramDO);
            }
        } else {

            // If we are adding a new AxisService param or overriding a param in the Configuration hierarchy
            if (p == null || !p.isLocked()) {
                axisService.addParameter(parameter);

                ServiceParameterDO param = new ServiceParameterDO();
                param.setName(parameter.getName());
                param.setValue(paramterElement.toString());
                param.setService(serviceDO);

                try {
                    pm.addEntity(param);
                } catch (DuplicateEntityException e) {
                    log.error("Service Parameter already exists", e);
                }
            }
        }
    }

    public void removeServiceSpecificParameter(String serviceId,
                                               String serviceVersion,
                                               OMElement paramterElement) throws AxisFault {

        AxisService axisService = getAxisConfig().getService(serviceId);

        if (axisService == null) {
            throw new AxisFault("Invalid service name '" + serviceId + "'");
        }

        // un-Persist the parameter
        ServiceDO serviceDO = pm.getService(serviceId,
                                            ServiceIdentifierDO.EMPTY_SERVICE_VERSION);
        Parameter parameter = ParameterUtil.createParameter(paramterElement);
        Parameter p = axisService.getParameter(parameter.getName());
        ServiceParameterDO paramDO = pm.getServiceParameter(serviceDO,
                                                            parameter.getName());
        if (p != null && paramDO != null) {

            // Setting a new value for a parameter declared in AxisService
            if (!p.isLocked()) {
                axisService.removeParameter(parameter);
                pm.deleteEntity(paramDO);
            }
        } else {

            if (p == null || !p.isLocked()) {
                axisService.removeParameter(parameter);


            }
        }
    }

    public void setServiceParameter2(String serviceName,
                                     Parameter parameter) throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceName);

        if (axisService == null) {
            throw new AxisFault("Invalid service name '" + serviceName + "'");
        }

        // Persist the parameter
        ServiceDO serviceDO = pm.getService(serviceName,
                                            ServiceIdentifierDO.EMPTY_SERVICE_VERSION);
        Parameter p = axisService.getParameter(parameter.getName());

        if (p != null) {
            if (!p.isLocked()) {
                axisService.addParameter(parameter);

                ServiceParameterDO paramDO = pm.getServiceParameter(serviceDO,
                                                                    parameter.getName());
                paramDO.setValue(p.getParameterElement().toString());
                pm.updateEntity(paramDO);
            }
        } else {
            axisService.addParameter(parameter);

            ServiceParameterDO param = new ServiceParameterDO();
            param.setName(parameter.getName());
            param.setValue(parameter.getParameterElement().toString());
            param.setService(serviceDO);

            try {
                pm.addEntity(param);
            } catch (DuplicateEntityException e) {
                log.error("Service Parameter already exists", e);
            }
        }
    }

    public void removeServiceParameter(String serviceName, String paramName) throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceName);
        axisService.removeParameter(new Parameter(paramName, null));
        ServiceDO serviceDO = pm.getService(serviceName, ServiceIdentifierDO.EMPTY_SERVICE_VERSION);
        ServiceParameterDO sp = new ServiceParameterDO();
        sp.setService(serviceDO);
        sp.setName(paramName);
        serviceDO.getParameters().remove(sp);
        pm.updateEntity(serviceDO);
    }

    public OMElement getWSDL(String serviceName) throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceName);

        if (axisService != null) {
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            String url = MessageContext.getCurrentMessageContext().getTo().getAddress();
            int ipindex = url.indexOf("//");
            String ip = null;

            if (ipindex >= 0) {
                ip = url.substring(ipindex + 2, url.length());

                int seperatorIndex = ip.indexOf(":");

                if (seperatorIndex > 0) {
                    ip = ip.substring(0, seperatorIndex);
                }
            }

            axisService.printWSDL(out, ip);

            try {
                XMLStreamReader xmlReader = XMLInputFactory.newInstance()
                        .createXMLStreamReader(new ByteArrayInputStream(
                                out.toByteArray()));

                OMFactory fac = OMAbstractFactory.getOMFactory();
                OMNamespace namespace = fac.createOMNamespace("http://org.apache.axis2/xsd",
                                                              "ns1");
                OMElement wsdlWrapper = fac.createOMElement("getWSDLResponse",
                                                            namespace);
                OMElement retvalue = fac.createOMElement("return", null);
                wsdlWrapper.addChild(retvalue);

                StAXOMBuilder staxOMBuilder = new StAXOMBuilder(fac, xmlReader);
                retvalue.addChild(staxOMBuilder.getDocumentElement());

                return wsdlWrapper;
            } catch (XMLStreamException e) {
                throw AxisFault.makeFault(e);
            }
        }

        return null;
    }

    public OMElement getSchema(String serviceName) throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceName);

        if (axisService != null) {
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            axisService.printSchema(out);

            try {
                XMLStreamReader xmlReader = XMLInputFactory.newInstance()
                        .createXMLStreamReader(new ByteArrayInputStream(
                                out.toByteArray()));
                OMFactory fac = OMAbstractFactory.getOMFactory();
                OMElement wsdlWrapper = fac.createOMElement("getWSDLReponse",
                                                            null);
                StAXOMBuilder staxOMBuilder = new StAXOMBuilder(fac, xmlReader);
                wsdlWrapper.addChild(staxOMBuilder.getDocumentElement());

                return wsdlWrapper;
            } catch (XMLStreamException e) {
                throw AxisFault.makeFault(e);
            }
        }

        return null;
    }

    public OMElement getPolicy(String serviceId, String serviceVersion) throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceId);

        if (axisService == null) {
            throw new AxisFault("invalid service name");
        }

        PolicyInclude servicePolicyInclude = axisService.getPolicyInclude();
        Policy servicePolicy = servicePolicyInclude.getPolicy();

        if (servicePolicy == null) {
            return PolicyUtil.getEmptyPolicyAsOMElement();
        }

        return PolicyUtil.getPolicyAsOMElement(servicePolicy);
    }

    public PolicyData[] getPolicies(String serviceId, String serviceVersion) throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceId);

        if (axisService == null) {
            throw new AxisFault("invalid service name");
        }

        ArrayList policyDataArray = new ArrayList();

        PolicyInclude servicePolicyInclude = axisService.getPolicyInclude();
        PolicyRegistry registry = servicePolicyInclude.getPolicyRegistry();

        List policyList;

        // services.xml
        policyList = servicePolicyInclude.getPolicyElements(PolicyInclude.AXIS_SERVICE_POLICY);

        if (!policyList.isEmpty()) {
            PolicyData policyData = new PolicyData();
            policyData.setWrapper("Policies in services.xml");
            policyData
                    .setPolycies(PolicyUtil.processPolicyElements(policyList.iterator(), registry));
            policyDataArray.add(policyData);
        }

        // wsdl:service
        policyList = servicePolicyInclude.getPolicyElements(PolicyInclude.SERVICE_POLICY);

        if (!policyList.isEmpty()) {
            PolicyData policyData = new PolicyData();
            policyData.setWrapper("Policies in wsdl:service");
            policyData
                    .setPolycies(PolicyUtil.processPolicyElements(policyList.iterator(), registry));
            policyDataArray.add(policyData);
        }

        // wsdl:portType
        policyList = servicePolicyInclude.getPolicyElements(PolicyInclude.PORT_TYPE_POLICY);

        if (!policyList.isEmpty()) {
            PolicyData policyData = new PolicyData();
            policyData.setWrapper("Policies in wsdl:portType");
            policyData
                    .setPolycies(PolicyUtil.processPolicyElements(policyList.iterator(), registry));
            policyDataArray.add(policyData);
        }

        // wsdl:port
        policyList = servicePolicyInclude.getPolicyElements(PolicyInclude.PORT_POLICY);

        if (!policyList.isEmpty()) {
            PolicyData policyData = new PolicyData();
            policyData.setWrapper("Policies in wsdl:port");
            policyData
                    .setPolycies(PolicyUtil.processPolicyElements(policyList.iterator(), registry));
            policyDataArray.add(policyData);
        }

        // wsdl:binding
        policyList = servicePolicyInclude.getPolicyElements(PolicyInclude.BINDING_POLICY);

        if (!policyList.isEmpty()) {
            PolicyData policyData = new PolicyData();
            policyData.setWrapper("Policies in wsdl:binding");
            policyData
                    .setPolycies(PolicyUtil.processPolicyElements(policyList.iterator(), registry));
            policyDataArray.add(policyData);
        }


        return (PolicyData[]) policyDataArray.toArray(new PolicyData[policyDataArray.size()]);
    }

    public boolean removePolicy(String serviceId,
                                String version,
                                String policyId) throws AxisFault {
        boolean isPolicyRemoved = false;
        AxisService axisService = getAxisConfig().getService(serviceId);
        if (axisService == null) {
            throw new AxisFault("Invalid service name " + serviceId);
        }
        PolicyInclude policyInclude = axisService.getPolicyInclude();
        if (policyInclude.getPolicy(policyId) != null) {
            policyInclude.removePolicyElement(policyId);
            isPolicyRemoved = true;
        }
        ServiceDO serviceDO = pm.getService(serviceId, version);
        List policiesToBeRemoved = new ArrayList();
        for (Iterator policyIter = serviceDO.getPolicies().iterator();
             policyIter.hasNext();) {
            ServicePolicyDO servicePolicyDO = (ServicePolicyDO) policyIter.next();
            if (servicePolicyDO.getUuid().equals(policyId)) {
                policiesToBeRemoved.add(servicePolicyDO);
            }
        }
        Set policies = serviceDO.getPolicies();
        for (Iterator iterator = policiesToBeRemoved.iterator(); iterator.hasNext();) {
            policies.remove(iterator.next());
            isPolicyRemoved = true;
        }
        pm.updateEntity(serviceDO);
        return isPolicyRemoved;
    }

    public void setPolicy(String serviceId,
                          String version,
                          String policyString) throws AxisFault {

        ByteArrayInputStream bais = new ByteArrayInputStream(policyString.getBytes());
        OMElement policyElement;
        try {
            policyElement = new StAXOMBuilder(bais).getDocumentElement();
            policyElement.build();
        } catch (Exception e) {
            throw new AxisFault("Cannot deserialize service policy. The policy may be invalid.", e);
        }

        AxisService axisService = getAxisConfig().getService(serviceId);
        if (axisService == null) {
            throw new AxisFault("Invalid service name " + serviceId);
        }

        if (!policyElement.getLocalName().equals(org.apache.neethi.Constants.ELEM_POLICY)) {
            policyElement = policyElement.getFirstElement();
        }

        OMAttribute attribute =
                policyElement.getAttribute(new QName(org.apache.neethi.Constants.URI_WSU_NS,
                                                     org.apache.neethi.Constants.ATTR_ID));

        if (attribute == null) {
            throw new AxisFault("ID Attribute not found in Policy element. " +
                                "Please verify that the policy XML is valid.");
        }
        String idAttrVal = attribute.getAttributeValue();

        PolicyInclude policyInclude = axisService.getPolicyInclude();

        Policy servicePolicy;
        try {
            servicePolicy = PolicyUtil.getPolicyFromOMElement(policyElement);

        } catch (NullPointerException npe) {
            /*
             * Since 'policyElement.toString()' is the actual string which is used to
             * create the policy object it logged instead of 'policyString'.
             */
            log.error("Error: When constructing Policy object from the string : "
                      + policyElement.toString(), npe);
            throw new AxisFault("Error: Invalid policy", npe);

        } catch (RuntimeException rte) {
            log.error("Error: When constructing Policy object from the string : "
                      + policyElement.toString(), rte);
            throw new AxisFault("Error: Invalid policy", rte);
        }

        ServicePolicyDO servicePolicyDO = new ServicePolicyDO();
        servicePolicyDO.setUuid(idAttrVal);
        servicePolicyDO.setService(pm.getService(serviceId,
                                                 ServiceIdentifierDO.EMPTY_SERVICE_VERSION));
        servicePolicyDO.setPolicy(policyElement.toString());

        if (policyInclude.getPolicy(idAttrVal) != null) {
            policyInclude.updatePolicy(servicePolicy);

            try {
                pm.updateServicePolicy(servicePolicyDO);
            } catch (ServicePolicyNotFoundException e) {
                throw AxisFault.makeFault(e);
            }

        } else {
            policyInclude.addPolicyElement(PolicyInclude.AXIS_SERVICE_POLICY, servicePolicy);
            servicePolicyDO.setType(PolicyInclude.AXIS_SERVICE_POLICY);

            try {
                pm.addServicePolicy(servicePolicyDO);
            } catch (ServicePolicyAlreadyExistsException e) {
                throw AxisFault.makeFault(e);
            }
        }
    }

    public String[] getServiceUsers(String serviceName) throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceName);

        if (axisService == null) {
            throw new AxisFault("invalid service name " + serviceName);
        }

        Collection users = new ArrayList();
        ServiceUserDO[] userDOs = pm.getUsersForService(serviceName,
                                                        ServiceIdentifierDO.EMPTY_SERVICE_VERSION);

        for (int i = 0; i < userDOs.length; i++) {
            ServiceUserDO userDO = userDOs[i];
            users.add(userDO.getUsername());
        }

        return (String[]) users.toArray(new String[users.size()]);
    }

    public ServiceUserData[] getServiceUsersWithDescription(String serviceName)
            throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceName);

        if (axisService == null) {
            throw new AxisFault("invalid service name " + serviceName);
        }

        ServiceUserDO[] userDOs = pm.getUsersForService(serviceName,
                                                        ServiceIdentifierDO.EMPTY_SERVICE_VERSION);

        ServiceUserData[] suData = new ServiceUserData[userDOs.length];

        if (userDOs.length == 0) {
            return new ServiceUserData[0];
        }

        ServiceUserDO serviceUserData;

        for (int i = 0; i < suData.length; i++) {
            serviceUserData = userDOs[i];

            ServiceUserData userData = new ServiceUserData();
            userData.setUser(serviceUserData.getUsername());
            userData.setDescription(serviceUserData.getDescription());
            suData[i] = userData;
        }

        return suData;
    }

    public ServiceRoleData[] getServiceRolesWithDescription(String serviceName)
            throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceName);

        if (axisService == null) {
            throw new AxisFault("invalid service name " + serviceName);
        }

        ServiceUserRoleDO[] roleDOs = pm.getRolesForService(serviceName,
                                                            ServiceIdentifierDO.EMPTY_SERVICE_VERSION);

        ServiceRoleData[] srData = new ServiceRoleData[roleDOs.length];

        if (roleDOs.length == 0) {
            return new ServiceRoleData[0];
        }

        ServiceUserRoleDO serviceRoleData;

        for (int i = 0; i < srData.length; i++) {
            serviceRoleData = roleDOs[i];

            ServiceRoleData roleData = new ServiceRoleData();
            roleData.setRole(serviceRoleData.getRole());
            roleData.setDescription(serviceRoleData.getDescription());
            srData[i] = roleData;
        }

        return srData;
    }

    /**
     * Get all the fully qualified class names of all the classes in this archive
     *
     * @param archiveId
     * @return all the fully qualified class names of all the classes in this archive
     * @throws AxisFault
     */
    public String[] getClassNames(String archiveId) throws AxisFault {
        String filePath = getFilePathFromArchiveId(archiveId);

        if (filePath != null) {
            try {
                String[] entries = new ArchiveManipulator().check(filePath);
                Collection classnames = new ArrayList();

                for (int i = 0; i < entries.length; i++) {
                    String entry = entries[i];

                    if (entry.endsWith(".class")) {
                        entry = entry.replace('/', '.').substring(0,
                                                                  entry.lastIndexOf(".class"));
                        classnames.add(entry);
                    }
                }

                return (String[]) classnames.toArray(new String[classnames.size()]);
            } catch (IOException e) {
                String msg = "Could not read archive";
                log.error(msg, e);
                throw new AxisFault(msg, e);
            }
        }

        return new String[]{""};
    }

    /**
     * This method will list all the public methods for the given classes.
     *
     * @param classNames full qualified class name
     * @return resourcesId array of resource Ids
     * @throws AxisFault will be thrown
     */
    public ClassMethodsData[] listMethodsForClass(String[] classNames, String[] resourcesId)
            throws AxisFault {
        if (classNames == null || classNames.length == 0) {
            String msg = "Cannot find classes";
            log.error(msg);
            throw new AxisFault(msg);
        }
        List methodExcludeList = new ArrayList();
        methodExcludeList.add("hashCode");
        methodExcludeList.add("getClass");
        methodExcludeList.add("equals");
        methodExcludeList.add("notify");
        methodExcludeList.add("notifyAll");
        methodExcludeList.add("toString");
        methodExcludeList.add("wait");
        List resourecesList = new ArrayList();
        for (int i = 0; i < resourcesId.length; i++) {
            String filePathFromArchiveId = getFilePathFromArchiveId(resourcesId[i]);
            if (filePathFromArchiveId == null) {
                throw new AxisFault("File cannot be located for id : " + resourcesId[i]);
            }
            try {
                resourecesList.add(new File(filePathFromArchiveId).toURL());
            } catch (MalformedURLException e) {
                String msg = "MalformedURLException";
                throw new AxisFault(msg, e);
            }
        }
        URL[] urls = (URL[]) resourecesList.toArray(new URL[resourecesList.size()]);
        ClassLoader classLoader =
                URLClassLoader.newInstance(urls, Thread.currentThread().getContextClassLoader());
        List classMethodList = new ArrayList();
        for (int i = 0; i < classNames.length; i++) {
            String className = classNames[i];
            try {
                Class clazz = classLoader.loadClass(className);
                ClassMethodsData classMethodsData = new ClassMethodsData();
                classMethodsData.setClassName(clazz.getName());
                Method[] methods = clazz.getMethods();
                List methodList = new ArrayList();
                for (int j = 0; j < methods.length; j++) {
                    Method method = methods[j];
                    String methodName = method.getName();
                    int modifiers = method.getModifiers();
                    if (Modifier.isPublic(modifiers) && !methodExcludeList.contains(methodName)) {
                        MethodData methodData = new MethodData();
                        methodData.setMethodName(methodName);
                        methodList.add(methodData);
                    }
                }
                findOverloadedMethods(methodList);
                classMethodsData.setMethods(
                        (MethodData[]) methodList.toArray(new MethodData[methodList.size()]));
                classMethodList.add(classMethodsData);
            } catch (ClassNotFoundException e) {
                String msg = "The class " + className + " cannot be loaded";
                log.error(msg);
            }

        }
        return (ClassMethodsData[]) classMethodList
                .toArray(new ClassMethodsData[classMethodList.size()]);

    }

    private void findOverloadedMethods(List methodList) {
        Map finalMap = new HashMap();
        int size = methodList.size();
        for (int k = 0; k < size - 1; k++) {
            MethodData startPointer = (MethodData) methodList.get(k);
            if (!finalMap.containsKey(startPointer.getMethodName())) {
                int count = 0;
                for (Iterator iterator = methodList.iterator(); iterator.hasNext();) {
                    MethodData method = (MethodData) iterator.next();
                    if (method.getMethodName().equals(startPointer.getMethodName())) {
                        count++;
                    }
                }
                for (Iterator iterator = methodList.iterator(); iterator.hasNext();) {
                    MethodData method = (MethodData) iterator.next();
                    if (method.getMethodName().equals(startPointer.getMethodName())) {
                        if (count > 1) {
                            method.setOverload(true);
                        }
                    }
                }
            }
        }

    }

    /**
     * Creates and deploys a service. This AAR will contain all the classe from the jar/zip file
     * corresponding to <code>archiveId</code>. In addition, a services.xml will be created, and all
     * of the <code>serviceClasses</code> will be added as services.
     *
     * @param archiveId  archive id
     * @param data       info array. data contains the excluded method names.
     * @param resourceId array of resource Ids that goes in the "lib" folder
     * @throws AxisFault will be thrown
     */
    public void createAndDeployService(String archiveId,
                                       ClassMethodsData[] data,
                                       String[] resourceId) throws AxisFault {
        String filePathFromArchiveId = getFilePathFromArchiveId(archiveId);

        if (filePathFromArchiveId == null) {
            String msg = "A non-existent file was requested";
            log.warn(msg);
            throw new AxisFault(msg);
        }

        int endIndex = filePathFromArchiveId.lastIndexOf(File.separator);
        String filePath = filePathFromArchiveId.substring(0, endIndex);
        String archiveFileName = filePathFromArchiveId.substring(endIndex);
        archiveFileName = archiveFileName.substring(0, archiveFileName.lastIndexOf("."));

        ArchiveManipulator archiveManipulator = new ArchiveManipulator();

        // ----------------- Unzip the file ------------------------------------
        String unzippeDir = filePath + File.separator + "temp";
        File unzipped = new File(unzippeDir);
        unzipped.mkdirs();

        try {
            archiveManipulator.extract(filePathFromArchiveId, unzippeDir);
        } catch (IOException e) {
            throw new AxisFault("Cannot extract archive", e);
        }

        //copy resources to lib folder.
        File resourceLibDir = new File(unzipped.getAbsolutePath() + File.separator + "lib");
        resourceLibDir.mkdirs();
        List resourceList = new ArrayList();
        for (int i = 0; i < resourceId.length; i++) {
            if (!resourceId[i].equals(archiveId)) {
                String fileAbsPath = getFilePathFromArchiveId(resourceId[i]);
                if (fileAbsPath == null) {
                    throw new AxisFault("Resource cannot be located");
                }
                resourceList.add(new File(fileAbsPath));
            }
        }
        FileManipulator fileMan = new FileManipulator();
        for (Iterator iterator = resourceList.iterator(); iterator.hasNext();) {
            File file = (File) iterator.next();
            try {
                String absPath = file.getAbsolutePath();
                if (File.separatorChar == '\\') {
                    absPath = absPath.replace("\\", "/");
                }
                int nameIndex = absPath.lastIndexOf("/");
                String fileName = absPath.substring(nameIndex + 1);
                fileMan.copyFile(file, new File(resourceLibDir, fileName));
            } catch (IOException e) {
                String msg = "Error occured when copying resources";
                throw new AxisFault(msg, e);
            }
        }

        // ---- Generate the services.xml and place it in META-INF -----
        File file = new File(unzippeDir + File.separator + "META-INF" +
                             File.separator + "services.xml");
        file.mkdirs();

        try {
            File absoluteFile = file.getAbsoluteFile();

            if (absoluteFile.exists()) {
                absoluteFile.delete();
            }

            absoluteFile.createNewFile();

            OutputStream os = new FileOutputStream(file);
            OMElement servicesXML = createServicesXML(data);
            servicesXML.build();
            servicesXML.serialize(os);
        } catch (Exception e) {
            String msg = "Cannot write services XML";
            log.error(msg, e);
            throw new AxisFault(msg, e);
        }

        // ----------------- Create the AAR ------------------------------------
        // These are the files to include in the ZIP file
        String outAARFilename = filePath + File.separator + archiveFileName +
                                ".aar";

        try {
            archiveManipulator.archiveDir(outAARFilename, unzipped.getPath());
        } catch (IOException e) {
            String msg = "Cannot create new AAR archive";
            log.error(msg, e);
            throw new AxisFault(msg, e);
        }

        //------------- Copy the AAR to the repository/services directory --------
        String repo = MessageContext.getCurrentMessageContext().getConfigurationContext()
                .getAxisConfiguration().getRepository()
                .getPath() + File.separator + "services";

        try {
            new FileManipulator().copyFile(new File(outAARFilename),
                                           new File(repo + File.separator + archiveFileName +
                                                    ".aar"));
        } catch (IOException e) {
            String msg = "Cannot copy AAR file to Repo";
            log.error(msg, e);
            throw new AxisFault(msg, e);
        }
    }

    private OMElement createServicesXMLFromSpringBeans(String[] springBeans,
                                                       String springContextLocation) {
        OMFactory factory = OMAbstractFactory.getOMFactory();
        OMNamespace emptyNS = factory.createOMNamespace("", "");
        OMElement serviceGroupEle = factory.createOMElement("serviceGroup", "", "");

        for (int i = 0; i < springBeans.length; i++) {
            String serviceBeanName = springBeans[i];


            OMElement serviceEle = factory.createOMElement("service", "", "");

            OMElement schemaEle = factory.createOMElement("schema", "", "");
            schemaEle.addAttribute(
                    factory.createOMAttribute("elementFormDefaultQualified", emptyNS, "false"));

            serviceEle.addAttribute(factory.createOMAttribute("name", emptyNS, serviceBeanName));

            OMElement msgReceiversEle = factory.createOMElement("messageReceivers",
                                                                "", "");
            OMElement msgReceiverEle1 = factory.createOMElement("messageReceiver",
                                                                "", "");
            msgReceiverEle1.addAttribute("mep",
                                         "http://www.w3.org/ns/wsdl/in-only", emptyNS);
            msgReceiverEle1.addAttribute("class",
                                         RPCInOnlyMessageReceiver.class.getName(),
                                         emptyNS);

            OMElement msgReceiverEle2 = factory.createOMElement("messageReceiver",
                                                                "", "");
            msgReceiverEle2.addAttribute("mep",
                                         "http://www.w3.org/ns/wsdl/in-out", emptyNS);
            msgReceiverEle2.addAttribute("class",
                                         RPCMessageReceiver.class.getName(),
                                         emptyNS);
            msgReceiversEle.addChild(msgReceiverEle1);
            msgReceiversEle.addChild(msgReceiverEle2);

            OMElement parameterEleServiceObjectSupplier =
                    factory.createOMElement("parameter", "", "");
            parameterEleServiceObjectSupplier.addAttribute("locked", "true", emptyNS);
            parameterEleServiceObjectSupplier
                    .addAttribute("name", "ServiceObjectSupplier", emptyNS);
            parameterEleServiceObjectSupplier.
                    setText(GenericApplicationContextSupplier.class.getName());

            OMElement parameterEleSpringBeanName = factory.createOMElement("parameter", "", "");
            parameterEleSpringBeanName.addAttribute("locked", "true", emptyNS);
            parameterEleSpringBeanName.addAttribute("name", "SpringBeanName", emptyNS);
            parameterEleSpringBeanName.setText(serviceBeanName);

            OMElement parameterEleSpringContextLocation =
                    factory.createOMElement("parameter", "", "");
            parameterEleSpringContextLocation.addAttribute("locked", "true", emptyNS);
            parameterEleSpringContextLocation
                    .addAttribute("name", "SpringContextLocation", emptyNS);
            parameterEleSpringContextLocation.setText(springContextLocation);

            serviceEle.addChild(schemaEle);
            serviceEle.addChild(msgReceiversEle);
            serviceEle.addChild(parameterEleServiceObjectSupplier);
            serviceEle.addChild(parameterEleSpringBeanName);
            serviceEle.addChild(parameterEleSpringContextLocation);
            serviceGroupEle.addChild(serviceEle);
        }
        return serviceGroupEle;
    }

    private OMElement createServicesXML(ClassMethodsData[] data) {
        OMFactory factory = OMAbstractFactory.getOMFactory();
        OMNamespace emptyNS = factory.createOMNamespace("", "");
        OMElement serviceGroupEle = factory.createOMElement("serviceGroup", "", "");

        for (int i = 0; i < data.length; i++) {
            ClassMethodsData datum = data[i];
            String serviceClass = datum.getClassName();
            String serviceName = serviceClass.substring(serviceClass.lastIndexOf(".") + 1);

            OMElement serviceEle = factory.createOMElement("service", "", "");

            OMElement schemaEle = factory.createOMElement("schema", "", "");
            schemaEle.addAttribute(
                    factory.createOMAttribute("elementFormDefaultQualified", emptyNS, "true"));

            serviceEle.addAttribute(factory.createOMAttribute("name", emptyNS,
                                                              serviceName));

            OMElement msgReceiversEle = factory.createOMElement("messageReceivers",
                                                                "", "");
            OMElement msgReceiverEle1 = factory.createOMElement("messageReceiver",
                                                                "", "");
            msgReceiverEle1.addAttribute("mep",
                                         "http://www.w3.org/ns/wsdl/in-only", emptyNS);
            msgReceiverEle1.addAttribute("class",
                                         "org.apache.axis2.rpc.receivers.RPCInOnlyMessageReceiver",
                                         emptyNS);

            OMElement msgReceiverEle2 = factory.createOMElement("messageReceiver",
                                                                "", "");
            msgReceiverEle2.addAttribute("mep",
                                         "http://www.w3.org/ns/wsdl/in-out", emptyNS);
            msgReceiverEle2.addAttribute("class",
                                         "org.apache.axis2.rpc.receivers.RPCMessageReceiver",
                                         emptyNS);
            msgReceiversEle.addChild(msgReceiverEle1);
            msgReceiversEle.addChild(msgReceiverEle2);

            OMElement parameterEle = factory.createOMElement("parameter", "", "");
            parameterEle.addAttribute("name", "ServiceClass", emptyNS);
            parameterEle.setText(serviceClass);

            serviceEle.addChild(schemaEle);
            serviceEle.addChild(msgReceiversEle);
            serviceEle.addChild(parameterEle);
            // Operations
            MethodData[] methodData = datum.getMethods();
            if (methodData != null) {
                if (methodData.length > 0) {
                    OMElement excludesEle = factory.createOMElement("excludeOperations", "", "");
                    serviceEle.addChild(excludesEle);
                    List operationDuplicationList = new ArrayList();
                    for (int j = 0; j < methodData.length; j++) {
                        String methodName = methodData[j].getMethodName();
                        if (!operationDuplicationList.contains(methodName)) {
                            OMElement operationEle = factory.createOMElement("operation", "", "");
                            operationDuplicationList.add(methodName);
                            operationEle.setText(methodName);
                            excludesEle.addChild(operationEle);
                        }
                    }
                }
            }
            serviceGroupEle.addChild(serviceEle);
        }

        return serviceGroupEle;
    }

    private String getFilePathFromArchiveId(String archiveId) {
        ConfigurationContext configCtx =
                MessageContext.getCurrentMessageContext().getConfigurationContext();
        Map fileResMap =
                (Map) configCtx.getProperty(org.wso2.wsas.ServerConstants.FILE_RESOURCE_MAP);

        return (String) fileResMap.get(archiveId);
    }

    public boolean deleteFaultyServiceArchive(String archiveName) throws AxisFault {
        if (File.separatorChar == '\\') {
            archiveName = archiveName.replace('/', '\\');
        }

        boolean isDeleted = false;
        if (archiveName != null && archiveName.trim().length() != 0) {
            File file = new File(archiveName);
            if (file.exists()) {
                if (!((file.isDirectory() && new FileManipulator().deleteDir(file)) ||
                      file.delete())) {
                    throw new AxisFault("Faulty service archive deletion failed. " +
                                        "Due to a JVM issue on MS-Windows, " +
                                        "AAR files cannot be deleted. " +
                                        "Please stop the server and manually delete this file.");
                } else {
                    isDeleted = true;
                    getAxisConfig().getFaultyServices().remove(archiveName);
                }
            }
        }

        return isDeleted;
    }

    public FaultService[] getFaultyServiceArchives() throws AxisFault {
        AxisConfiguration ac = getAxisConfig();
        Hashtable faultyServices = ac.getFaultyServices();
        ArrayList fsList = new ArrayList();

        for (Iterator keys = faultyServices.keySet().iterator();
             keys.hasNext();) {
            String key = (String) keys.next();
            String fault = (String) faultyServices.get(key);
            FaultService fs = new FaultService();

            if (File.separatorChar == '\\') {
                key = key.replace('\\', '/');
            }

            fs.setServiceName(key);
            fs.setFault(fault);
            fsList.add(fs);
        }

        return (FaultService[]) fsList.toArray(new FaultService[fsList.size()]);
    }

    public int getNumberOfFaultyServices() throws AxisFault {
        return getAxisConfig().getFaultyServices().size();
    }

    public SpringBeansData getSpringBeanNames(String springContextId,
                                              String springBeanId) throws AxisFault {
        //Manipulation of springContext to ${RepositoryLocation}/spring

        String springContextFilePath = getFilePathFromArchiveId(springContextId);
        String springBeanFilePath = getFilePathFromArchiveId(springBeanId);
        SpringBeansData data = new SpringBeansData();
        data.setSpringContext(springContextId);
        try {
            ApplicationContext aCtx =
                    GenericApplicationContextUtil
                            .getSpringApplicationContext(springContextFilePath, springBeanFilePath);
            String[] beanDefintions = aCtx.getBeanDefinitionNames();
            data.setBeans(beanDefintions);
        } catch (Exception e) {
            throw new AxisFault("Cannot load Spring beans. Please check the Spring context " +
                                "configuration file and verify that the defined Spring beans exist" +
                                " in the .jar file.");
        }
        return data;
    }

    public void createAndDeploySpringBean(String springContextId,
                                          String springBeanId,
                                          String[] beanClasses) throws AxisFault {
        String filePathFromArchiveId = getFilePathFromArchiveId(springBeanId);
        String filePathForSpringContext = getFilePathFromArchiveId(springContextId);
        ConfigurationContext configCtx =
                MessageContext.getCurrentMessageContext().getConfigurationContext();
        Map fileResMap =
                (Map) configCtx.getProperty(org.wso2.wsas.ServerConstants.FILE_RESOURCE_MAP);
        fileResMap.remove(springContextId);

        if (filePathFromArchiveId == null) {
            String msg = "A non-existent file was requested";
            log.warn(msg);
            throw new AxisFault(msg);
        }

        int endIndex = filePathFromArchiveId.lastIndexOf(File.separator);
        String filePath = filePathFromArchiveId.substring(0, endIndex);
        String archiveFileName = filePathFromArchiveId.substring(endIndex);
        archiveFileName = archiveFileName.substring(0,
                                                    archiveFileName.lastIndexOf("."));

        ArchiveManipulator archiveManipulator = new ArchiveManipulator();

        // ----------------- Unzip the file ------------------------------------
        String unzippeDir = filePath + File.separator + "springTemp";
        File unzipped = new File(unzippeDir);
        unzipped.mkdirs();

        try {
            archiveManipulator.extract(filePathFromArchiveId, unzippeDir);
        } catch (IOException e) {
            throw new AxisFault("Cannot extract archive", e);
        }

        //TODO copy the spring xml
        String springContextRelLocation = "spring/context.xml";
        try {
            File springContextRelDir =
                    new File(unzippeDir + File.separator + "spring");
            springContextRelDir.mkdirs();
            File absFile = new File(springContextRelDir, "context.xml");
            if (!absFile.exists()) {
                absFile.createNewFile();
            }
            File file = new File(filePathForSpringContext);
            FileInputStream in = new FileInputStream(file);
            FileOutputStream out = new FileOutputStream(absFile);
            // Transfer bytes from in to out
            byte[] buf = new byte[1024];
            int len;
            while ((len = in.read(buf)) > 0) {
                out.write(buf, 0, len);
            }
            in.close();
            out.close();

        } catch (FileNotFoundException e) {
            throw AxisFault.makeFault(e);
        } catch (IOException e) {
            throw AxisFault.makeFault(e);
        }

        // ---- Generate the services.xml and place it in META-INF -----
        File file = new File(unzippeDir + File.separator + "META-INF" +
                             File.separator + "services.xml");
        file.mkdirs();

        try {
            File absoluteFile = file.getAbsoluteFile();

            if (absoluteFile.exists()) {
                absoluteFile.delete();
            }

            absoluteFile.createNewFile();

            OutputStream os = new FileOutputStream(file);
            OMElement servicesXML =
                    createServicesXMLFromSpringBeans(beanClasses, springContextRelLocation);
            servicesXML.build();
            servicesXML.serialize(os);
        } catch (Exception e) {
            String msg = "Cannot write services XML";
            log.error(msg, e);
            throw new AxisFault(msg, e);
        }

        // ----------------- Create the AAR ------------------------------------
        // These are the files to include in the ZIP file
        String outAARFilename = filePath + File.separator + archiveFileName + ".aar";

        try {
            archiveManipulator.archiveDir(outAARFilename, unzipped.getPath());
        } catch (IOException e) {
            String msg = "Cannot create new AAR archive";
            log.error(msg, e);
            throw new AxisFault(msg, e);
        }

        //------------- Copy the AAR to the respository/services directory --------
        String repo = MessageContext.getCurrentMessageContext().getConfigurationContext()
                .getAxisConfiguration().getRepository()
                .getPath() + File.separator + "services";

        try {
            new FileManipulator().copyFile(new File(outAARFilename),
                                           new File(repo + File.separator + archiveFileName +
                                                    ".aar"));
        } catch (IOException e) {
            String msg = "Cannot copy AAR file to Repo";
            log.error(msg, e);
            throw new AxisFault(msg, e);
        }
    }

    public void createAndDeployEJBService(String archiveId,
                                          String[] serviceClasses,
                                          String jnpProviderUrl,
                                          String beanJNDIName,
                                          String homeInterface,
                                          String remoteInterface) throws AxisFault {
        //Find details of selected ejb application server configuration
        EJBAppServerDO ejbAppServerDO = pm.getEJBAppServerConfiguration(jnpProviderUrl);
        if (ejbAppServerDO == null) {
            throw new AxisFault("Non-existance Application server configuration");
        }

        //save ejb configuration
        EJBProviderAdmin ejbProviderAdmin = new EJBProviderAdmin();

        //search for existing configuration, if found abort service deployment
        try {
            EJBProviderDO ejbProviderDO = ejbProviderAdmin.getEJBConfiguration(beanJNDIName,
                                                                               jnpProviderUrl);
            if (!(ejbProviderDO == null)) {
                //configuration found.Throw exception
                throw new AxisFault("A Service exists for the provided JNDI name(" +
                                    beanJNDIName + ") and JNP Url(" + jnpProviderUrl + ").");
            }
        } catch (Exception e) {
            throw new AxisFault("A Service exists for the provided JNDI name(" +
                                beanJNDIName + ") and JNP Url(" + jnpProviderUrl + ").");
        }


        ejbProviderAdmin.addEJBConfiguration(ejbAppServerDO.getProviderURL()
                , ejbAppServerDO.getJndiContextClass()
                , ejbAppServerDO.getUserName()
                , ejbAppServerDO.getUserName()
                , beanJNDIName
                , homeInterface
                , remoteInterface
                , ejbAppServerDO.getAppServerType());

        String filePathFromArchiveId = getFilePathFromArchiveId(archiveId);

        if (filePathFromArchiveId == null) {
            String msg = "A non-existent file was requested";
            log.warn(msg);
            throw new AxisFault(msg);
        }

        int endIndex = filePathFromArchiveId.lastIndexOf(File.separator);
        String filePath = filePathFromArchiveId.substring(0, endIndex);
        String archiveFileName = filePathFromArchiveId.substring(endIndex);
        archiveFileName = archiveFileName.substring(0,
                                                    archiveFileName.lastIndexOf("."));

        ArchiveManipulator archiveManipulator = new ArchiveManipulator();

        //----------------- Unzip the file ------------------------------------
        String unzippeDir = filePath + File.separator + "temp";
        File unzipped = new File(unzippeDir);
        unzipped.mkdirs();

        try {
            archiveManipulator.extract(filePathFromArchiveId, unzippeDir);
        } catch (IOException e) {
            throw new AxisFault("Cannot extract archive", e);
        }

        //---- Generate the services.xml and place it in META-INF -----
        File file = new File(unzippeDir + File.separator + "META-INF" +
                             File.separator + "services.xml");
        file.mkdirs();

        try {
            File absoluteFile = file.getAbsoluteFile();
            if (absoluteFile.exists()) {
                absoluteFile.delete();
            }
            absoluteFile.createNewFile();
            OutputStream os = new FileOutputStream(file);
            OMElement servicesXML = createServicesXMLForEJBService(serviceClasses,
                                                                   ejbAppServerDO.getProviderURL(),
                                                                   ejbAppServerDO.getJndiContextClass(),
                                                                   ejbAppServerDO.getUserName(),
                                                                   ejbAppServerDO.getPassword(),
                                                                   beanJNDIName,
                                                                   homeInterface,
                                                                   remoteInterface);
            servicesXML.build();
            servicesXML.serialize(os);
        } catch (Exception e) {
            String msg = "Cannot write services XML";
            log.error(msg, e);
            throw new AxisFault(msg, e);
        }

        //----------------- Create the AAR ------------------------------------
        //These are the files to include in the ZIP file
        String outAARFilename = filePath + File.separator + archiveFileName +
                                ".aar";

        try {
            archiveManipulator.archiveDir(outAARFilename, unzipped.getPath());
        } catch (IOException e) {
            String msg = "Cannot create new AAR archive";
            log.error(msg, e);
            throw new AxisFault(msg, e);
        }

        //------------- Copy the AAR to the respository/services directory --------
        String repo = MessageContext.getCurrentMessageContext().getConfigurationContext()
                .getAxisConfiguration().getRepository()
                .getPath() + File.separator + "services";

        try {
            new FileManipulator().copyFile(new File(outAARFilename),
                                           new File(repo + File.separator + archiveFileName +
                                                    ".aar"));
        } catch (IOException e) {
            String msg = "Cannot copy AAR file to Repo";
            log.error(msg, e);
            throw new AxisFault(msg, e);
        }
    }


    /*
	<serviceGroup>
       <service name="TimeService">
               <description>Time!! web service</description>
               <parameter name="ServiceClass" locked="false">org.apache.axis2.samples.ejb.session.interfaces.TimeServiceRemote</parameter>
               <messageReceivers>
                       <messageReceiver mep="http://www.w3.org/2004/08/wsdl/in-only" class="org.apache.axis2.rpc.receivers.ejb.EJBInOnlyMessageReceiver"/>
                       <messageReceiver mep="http://www.w3.org/2004/08/wsdl/in-out" class="org.apache.axis2.rpc.receivers.ejb.EJBMessageReceiver"/>
               </messageReceivers>
               <parameter name="remoteInterfaceName">org.apache.axis2.samples.ejb.session.interfaces.TimeServiceRemote</parameter>
               <parameter name="homeInterfaceName">org.apache.axis2.samples.ejb.session.interfaces.TimeServiceRemoteHome</parameter>
               <parameter name="beanJndiName">org.apache.axis2.samples.ejb.session.interfaces.TimeServiceLocalHome</parameter>
               <parameter name="service_type">ejb_service</parameter>
               <!--for jboss -->
               <parameter name="providerUrl">jnp://localhost:1099</parameter>
               <parameter name="jndiContextClass">org.jnp.interfaces.NamingContextFactory</parameter>

               <!--for geronimo
               <parameter name="providerUrl">localhost:4201</parameter>
               <parameter name="jndiContextClass">org.openejb.client.JNDIContext</parameter>
               -->
       </service>
	</serviceGroup>
    */
    private OMElement createServicesXMLForEJBService(String[] serviceClasses, String providerURL,
                                                     String jndiContextClass,
                                                     String userName,
                                                     String password,
                                                     String beanJNDIName,
                                                     String homeInterface,
                                                     String remoteInterface) {

        OMFactory factory = OMAbstractFactory.getOMFactory();
        OMNamespace emptyNS = factory.createOMNamespace("", "");
        OMElement serviceGroupEle = factory.createOMElement("serviceGroup", "", "");

        //Although we have a loop here, only one class will come
        for (int i = 0; i < serviceClasses.length; i++) {
            String serviceClass = serviceClasses[i];
            String serviceName = serviceClass.substring(serviceClass.lastIndexOf(
                    ".") + 1);

            OMElement serviceEle = factory.createOMElement("service", "", "");

            OMElement schemaEle = factory.createOMElement("schema", "", "");
            schemaEle.addAttribute(
                    factory.createOMAttribute("elementFormDefaultQualified", emptyNS, "false"));

            serviceEle.addAttribute(factory.createOMAttribute("name", emptyNS,
                                                              serviceName));

            OMElement msgReceiversEle = factory.createOMElement("messageReceivers",
                                                                "", "");
            OMElement msgReceiverEle1 = factory.createOMElement("messageReceiver",
                                                                "", "");
            msgReceiverEle1.addAttribute("mep",
                                         "http://www.w3.org/ns/wsdl/in-only", emptyNS);
            msgReceiverEle1.addAttribute("class",
                                         "org.apache.axis2.rpc.receivers.ejb.EJBInOnlyMessageReceiver",
                                         emptyNS);

            OMElement msgReceiverEle2 = factory.createOMElement("messageReceiver",
                                                                "", "");
            msgReceiverEle2.addAttribute("mep",
                                         "http://www.w3.org/ns/wsdl/in-out", emptyNS);
            msgReceiverEle2.addAttribute("class",
                                         "org.apache.axis2.rpc.receivers.ejb.EJBMessageReceiver",
                                         emptyNS);
            msgReceiversEle.addChild(msgReceiverEle1);
            msgReceiversEle.addChild(msgReceiverEle2);

            OMElement parameterEle1 = factory.createOMElement("parameter", "", "");
            parameterEle1.addAttribute("name", EJBUtil.EJB_REMOTE_INTERFACE_NAME, emptyNS);
            parameterEle1.setText(remoteInterface);

            OMElement parameterEle2 = factory.createOMElement("parameter", "", "");
            parameterEle2.addAttribute("name", EJBUtil.EJB_HOME_INTERFACE_NAME, emptyNS);
            parameterEle2.setText(homeInterface);

            OMElement parameterEle3 = factory.createOMElement("parameter", "", "");
            parameterEle3.addAttribute("name", EJBUtil.EJB_JNDI_NAME, emptyNS);
            parameterEle3.setText(beanJNDIName);

            OMElement parameterEle4 = factory.createOMElement("parameter", "", "");
            parameterEle4.addAttribute("name", EJBUtil.EJB_PROVIDER_URL, emptyNS);
            parameterEle4.setText(providerURL);

            OMElement parameterEle5 = factory.createOMElement("parameter", "", "");
            parameterEle5.addAttribute("name", EJBUtil.EJB_INITIAL_CONTEXT_FACTORY, emptyNS);
            parameterEle5.setText(jndiContextClass);

            OMElement parameterEle6 = factory.createOMElement("parameter", "", "");
            parameterEle6.addAttribute("name", EJBUtil.EJB_JNDI_USERNAME, emptyNS);
            parameterEle6.setText(userName);

            OMElement parameterEle7 = factory.createOMElement("parameter", "", "");
            parameterEle7.addAttribute("name", EJBUtil.EJB_JNDI_PASSWORD, emptyNS);
            parameterEle7.setText(password);

            OMElement parameterEle8 = factory.createOMElement("parameter", "", "");
            parameterEle8.addAttribute("name", "ServiceClass", emptyNS);
            parameterEle8.setText(serviceClass);

            OMElement parameterEle9 = factory.createOMElement("parameter", "", "");
            parameterEle9.addAttribute("name", ServerConstants.SERVICE_TYPE, emptyNS);
            parameterEle9.setText(ServerConstants.SERVICE_TYPE_EJB);

            serviceEle.addChild(schemaEle);
            serviceEle.addChild(msgReceiversEle);
            serviceEle.addChild(parameterEle1);
            serviceEle.addChild(parameterEle2);
            serviceEle.addChild(parameterEle3);
            serviceEle.addChild(parameterEle4);
            serviceEle.addChild(parameterEle5);
            serviceEle.addChild(parameterEle6);
            serviceEle.addChild(parameterEle7);
            serviceEle.addChild(parameterEle8);
            serviceEle.addChild(parameterEle9);

            serviceGroupEle.addChild(serviceEle);
        }
        return serviceGroupEle;
    }

}
