/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.admin.service;

import org.apache.axis2.AxisFault;
import org.apache.axis2.context.ConfigurationContext;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.wso2.utils.AbstractAdmin;
import org.wso2.utils.MBeanRegistrar;
import org.wso2.utils.ServerConfiguration;
import org.wso2.wsas.ServerConstants;
import org.wso2.wsas.ServerManagement;
import org.wso2.wsas.admin.service.util.ServerData;
import org.wso2.wsas.admin.service.util.ServerStatus;
import org.wso2.wsas.util.Controllable;

import java.util.HashMap;


/**
 * Admin service to manage server related operations
 */
public class ServerAdmin extends AbstractAdmin implements ServerAdminMBean {
    private static final Log log = LogFactory.getLog(ServerAdmin.class);
    private Controllable controllable;

    static {
        MBeanRegistrar.registerMBean(new ServerAdmin());
    }

    public ServerAdmin() {
    }

    public ServerData getServerData() throws AxisFault {
        return new ServerData(ServerConstants.SERVER_NAME,
                              getAxisConfig().getRepository().toString());
    }

    public String getServerDataAsString() throws Exception {
        try {
            return getServerData().toString();
        } catch (AxisFault e) {
            String msg = "Cannot get server data";
            log.error(msg, e);
            throw new Exception(msg);
        }
    }

    public String getServerVersion() {
        return ServerConfiguration.getInstance().getFirstProperty("Version");
    }

    public ServerStatus getStatus() throws AxisFault {
        ServerStatus serverStatus = new ServerStatus();
        serverStatus.update(getAxisConfig());
        return serverStatus;
    }

    public void restart() {
        ConfigurationContext configurationContext = getConfigContext();
        controllable =
                (Controllable) configurationContext.
                        getProperty(ServerConstants.WSO2WSAS_INSTANCE);
        Thread th = new Thread() {
            public void run() {
                try {
                    Thread.sleep(1000);
                    controllable.restart();
                } catch (Exception e) {
                    String msg = "Cannot restart server";
                    log.error(msg, e);
                    throw new RuntimeException(msg, e);
                }
            }
        };
        th.start();
    }

    public void restartGracefully() {
        ConfigurationContext configurationContext = getConfigContext();
        controllable =
                (Controllable) configurationContext.
                        getProperty(ServerConstants.WSO2WSAS_INSTANCE);
        Thread th = new Thread() {
            public void run() {
                try {
                    Thread.sleep(1000);
                    controllable.restartGracefully();
                } catch (Exception e) {
                    String msg = "Cannot restart server";
                    log.error(msg, e);
                    throw new RuntimeException(msg, e);
                }
            }
        };
        th.start();
    }

    public void shutdown() {
        ConfigurationContext configurationContext = getConfigContext();
        controllable =
                (Controllable) configurationContext.
                        getProperty(ServerConstants.WSO2WSAS_INSTANCE);
        Thread th = new Thread() {
            public void run() {
                try {
                    Thread.sleep(1000);
                    controllable.shutdown();
                } catch (Exception e) {
                    String msg = "Cannot shutdown server";
                    log.error(msg, e);
                    throw new RuntimeException(msg, e);
                }
            }
        };
        th.start();
    }

    public void shutdownGracefully() {
        ConfigurationContext configurationContext = getConfigContext();
        controllable =
                (Controllable) configurationContext.
                        getProperty(ServerConstants.WSO2WSAS_INSTANCE);
        Thread th = new Thread() {
            public void run() {
                try {
                    Thread.sleep(1000);
                    controllable.shutdownGracefully();
                } catch (Exception e) {
                    String msg = "Cannot gracefully shutdown server";
                    log.error(msg, e);
                    throw new RuntimeException(msg, e);
                }
            }
        };
        th.start();
    }

    public void startMaintenance() throws Exception {
        try {
            org.wso2.wsas.ServerStatus.setServerInMaintenance();
        } catch (AxisFault e) {
            String msg = "Cannot set server to maintenance mode";
            log.error(msg, e);
        }
        HashMap inTransports = getAxisConfig().getTransportsIn();
        new ServerManagement(inTransports).startMaintenance();
    }

    public void endMaintenance() throws Exception {
        HashMap inTransports = getAxisConfig().getTransportsIn();
        new ServerManagement(inTransports).endMaintenance();
        try {
            org.wso2.wsas.ServerStatus.setServerRunning();
        } catch (AxisFault e) {
            String msg = "Cannot set server to running mode";
            log.error(msg, e);
        }
    }

    public boolean isAlive() {
        return true;
    }

    public String getServerStatus() throws Exception {
        try {
            return org.wso2.wsas.ServerStatus.getCurrentStatus();
        } catch (AxisFault e) {
            String msg = "Could not get current server status";
            log.error(msg);
            throw new Exception(msg, e);
        }
    }
}
