/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.admin.service;

import org.apache.axiom.om.OMAttribute;
import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.apache.axiom.om.xpath.AXIOMXPath;
import org.apache.axis2.AxisFault;
import org.apache.axis2.description.AxisModule;
import org.apache.axis2.description.AxisService;
import org.apache.axis2.description.PolicyInclude;
import org.apache.axis2.engine.AxisConfiguration;
import org.apache.commons.io.output.ByteArrayOutputStream;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.neethi.Policy;
import org.apache.neethi.PolicyEngine;
import org.apache.rampart.policy.RampartPolicyBuilder;
import org.apache.rampart.policy.RampartPolicyData;
import org.apache.rampart.policy.model.CryptoConfig;
import org.apache.rampart.policy.model.RampartConfig;
import org.apache.ws.secpolicy.model.SecureConversationToken;
import org.apache.ws.secpolicy.model.Token;
import org.apache.ws.security.handler.WSHandlerConstants;
import org.jaxen.SimpleNamespaceContext;
import org.jaxen.XPath;
import org.wso2.utils.AbstractAdmin;
import org.wso2.wsas.ServerConstants;
import org.wso2.wsas.admin.service.util.SecurityAssignment;
import org.wso2.wsas.admin.service.util.SelectableString;
import org.wso2.wsas.admin.service.util.UtKeystoreSecurityAssignment;
import org.wso2.wsas.persistence.PersistenceManager;
import org.wso2.wsas.persistence.dataobject.KeyStoreDO;
import org.wso2.wsas.persistence.dataobject.ModuleDO;
import org.wso2.wsas.persistence.dataobject.SecurityScenarioDO;
import org.wso2.wsas.persistence.dataobject.ServiceDO;
import org.wso2.wsas.persistence.dataobject.ServiceIdentifierDO;
import org.wso2.wsas.persistence.dataobject.ServiceUserDO;
import org.wso2.wsas.persistence.dataobject.ServiceUserRoleDO;
import org.wso2.wsas.persistence.exception.ServiceNotFoundException;
import org.wso2.wsas.security.ServerCrypto;
import org.wso2.wsas.security.WSS4JUtil;
import org.wso2.wsas.security.util.RampartConfigUtil;
import org.wso2.wsas.trust.SecurityTokenStore;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;
import java.io.File;
import java.io.FileFilter;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

/**
 * Admin service for configuring Security scenarios
 */
public class SecurityScenarioConfigAdmin extends AbstractAdmin {

    public static final String USER = "rampart.config.user";
    public static final String ENCRYPTION_USER = "rampart.config.encryption.user";
    public static final Log log = LogFactory.getLog(SecurityScenarioConfigAdmin.class);

    private PersistenceManager pm = new PersistenceManager();
    private String scenarioDir =
            System.getProperty(ServerConstants.WSO2WSAS_HOME) + File.separator + "conf" +
            File.separator + "rampart" + File.separator;
    private static final String WS_SEC_UTILITY_NS =
            "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd";
    private static final QName ID = new QName(WS_SEC_UTILITY_NS, "Id");
    private static final String WS_POLICY_NS = "http://schemas.xmlsoap.org/ws/2004/09/policy";
    private static final String SCENARIO_DISABLE_SECURITY = "DisableSecurity";

    public final static int DEFAULT_TS_TTL = 300;
    public final static int DEFAULT_TS_MAX_SKEW = 300;

    public SecurityScenarioConfigAdmin() {
    }

    public SecurityScenarioDO[] getScenarios(String serviceName) {
        SecurityScenarioDO[] scenarios = pm.getAllSecurityScenarios();
        ServiceDO service = new ServiceDO();
        ServiceIdentifierDO serviceIdentifierDO = new ServiceIdentifierDO();
        serviceIdentifierDO.setServiceId(serviceName);
        service.setServiceIdentifierDO(serviceIdentifierDO);
        boolean isServiceScenarioFound = false;
        for (int i = 0; i < scenarios.length; i++) {
            SecurityScenarioDO scenario = scenarios[i];
            if (scenario.services.contains(service)) {
                scenario.setIsCurrentScenario(true);
                isServiceScenarioFound = true;
                break; // a service can have only one current scenario
            }
        }
        if (!isServiceScenarioFound) {
            for (int i = 0; i < scenarios.length; i++) {
                SecurityScenarioDO scenario = scenarios[i];
                if (scenario.getScenarioId().equalsIgnoreCase(SCENARIO_DISABLE_SECURITY)) {
                    scenario.setIsCurrentScenario(true);
                    break;
                }
            }
        }
        return scenarios;
    }

    private void configureScenario(String serviceName,
                                   String scenarioId,
                                   Properties props) throws AxisFault {
        disengageModules(serviceName, SCENARIO_DISABLE_SECURITY);

        RampartConfig rampartConfig = new RampartConfig();
        rampartConfig.setTokenStoreClass(SecurityTokenStore.class.getName());

        populateRampartConfig(rampartConfig, props);

        ServiceAdmin serviceAdmin = new ServiceAdmin();

        // Remove policies of other scenarios
        removeAllSecurityScenarioPolicies(serviceAdmin, serviceName);

        // Add the policy to the axis service
        String scenarioPolicyXml = scenarioDir + scenarioId + "-policy.xml";
        try {
            Policy policy = loadPolicy(scenarioPolicyXml);
            if (rampartConfig != null) {
                policy.addAssertion(rampartConfig);
            }
            AxisService axisService = getAxisConfig().getService(serviceName);
            axisService.getPolicyInclude().addPolicyElement(PolicyInclude.AXIS_SERVICE_POLICY,
                                                            policy);

            // When there is a transport binding sec policy assertion,
            // the service should be exposed only via HTTPS
            Iterator alternatives = policy.getAlternatives();
            if (alternatives.hasNext()) {
                List it = (List) alternatives.next();
                boolean httpsRequired = false;

                RampartPolicyData rampartPolicyData = RampartPolicyBuilder.build(it);
                if (rampartPolicyData.isTransportBinding()) {
                    httpsRequired = true;
                } else if (rampartPolicyData.isSymmetricBinding()) {
                    Token encrToken = rampartPolicyData.getEncryptionToken();
                    if (encrToken instanceof SecureConversationToken) {
                        Policy bsPol = ((SecureConversationToken) encrToken).getBootstrapPolicy();
                        Iterator alts = bsPol.getAlternatives();
                        if (alts.hasNext()) {
                        }
                        List bsIt = (List) alts.next();
                        RampartPolicyData bsRampartPolicyData = RampartPolicyBuilder.build(bsIt);
                        httpsRequired = bsRampartPolicyData.isTransportBinding();
                    }
                }

                if (httpsRequired) {
                    List httpsTransports = new ArrayList();
                    for (Iterator iter = getAxisConfig().getTransportsIn().keySet().iterator();
                         iter.hasNext();) {
                        String transport = (String) iter.next();
                        if (transport.toLowerCase().indexOf(ServerConstants.HTTPS_TRANSPORT) != -1) {
                            httpsTransports.add(transport);
                        }
                    }
                    serviceAdmin.
                            exposeServiceOnlyViaTransports(serviceName,
                                                           (String[]) httpsTransports.
                                                                   toArray(new String[httpsTransports.size()]));

                }
            }

            //Add the callback handler instance
            if (axisService.getParameter(WSHandlerConstants.PW_CALLBACK_REF) == null) {
                axisService.addParameter(WSS4JUtil.
                        getPasswordCallBackRefParameter(serviceName,
                                                        ServiceIdentifierDO.EMPTY_SERVICE_VERSION));
            }

            // Persist the policy
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            XMLStreamWriter xmlStreamWriter =
                    XMLOutputFactory.newInstance().createXMLStreamWriter(baos);
            policy.serialize(xmlStreamWriter);
            xmlStreamWriter.flush();
            removeAllSecurityScenarioPolicies(serviceAdmin, serviceName);
            serviceAdmin.setPolicy(serviceName,
                                   ServiceIdentifierDO.EMPTY_SERVICE_VERSION,
                                   baos.toString());
        } catch (Exception e) {
            throw new AxisFault("Could not load security policy", e);
        }

        // persist the updated scenario after adding the relevant service
        ServiceDO service = pm.getService(serviceName, ServiceIdentifierDO.EMPTY_SERVICE_VERSION);
        SecurityScenarioDO securityScenario = pm.getSecurityScenario(scenarioId);
        if (service != null && securityScenario != null) {
            securityScenario.addService(service);
            pm.updateEntity(securityScenario);
            pm.updateEntity(service);
        }
    }

    private static Policy loadPolicy(String policyXMLFile) throws FileNotFoundException,
                                                                  XMLStreamException {
        return PolicyEngine.getPolicy(new StAXOMBuilder(policyXMLFile).getDocumentElement());
    }

    public void disableSecurity(String serviceName) throws AxisFault {
        ServiceAdmin serviceAdmin = new ServiceAdmin();
        removeAllSecurityScenarioPolicies(serviceAdmin, serviceName);
        ServiceDO serviceDO = pm.getService(serviceName, ServiceIdentifierDO.EMPTY_SERVICE_VERSION);
        serviceDO.setIsUTAuthEnabled(false);
        serviceDO.setIsExposedOnAllTransports(true);
        getAxisConfig().getService(serviceName).setEnableAllTransports(true);
        SecurityScenarioDO securityScenario = serviceDO.getSecurityScenario();
        if (securityScenario != null) {
            securityScenario.removeService(serviceDO);

            // Remove all security related stuff
            serviceDO.removeAllTrustedCertStores();
            serviceDO.setPrivateKeyStore(null);

            for (Iterator iter = serviceDO.getUsers().iterator(); iter.hasNext();) {
                ServiceUserDO userDO = (ServiceUserDO) iter.next();
                userDO.getServices().remove(serviceDO);
                pm.updateEntity(userDO);
            }
            for (Iterator iter = serviceDO.getRoles().iterator(); iter.hasNext();) {
                ServiceUserRoleDO roleDO = (ServiceUserRoleDO) iter.next();
                roleDO.getServices().remove(serviceDO);
                pm.updateEntity(roleDO);
            }
            serviceDO.removeAllRoles();
            serviceDO.removeAllUsers();
            pm.updateEntity(serviceDO);

            disengageModules(serviceName, SCENARIO_DISABLE_SECURITY);
        }
    }

    private void removeAllSecurityScenarioPolicies(ServiceAdmin serviceAdmin,
                                                   String serviceName) throws AxisFault {
        File scDir = new File(scenarioDir);
        File[] policyFiles = scDir.listFiles(new FileFilter() {
            public boolean accept(File pathname) {
                return pathname.getAbsolutePath().endsWith("-policy.xml");
            }
        });
        for (int i = 0; i < policyFiles.length; i++) {
            File policyFile = policyFiles[i];
            try {
                StAXOMBuilder builder = new StAXOMBuilder(policyFile.getAbsolutePath());
                SimpleNamespaceContext nsCtx = new SimpleNamespaceContext();
                nsCtx.addNamespace("wsp", WS_POLICY_NS);
                nsCtx.addNamespace("wsu", WS_SEC_UTILITY_NS);
                XPath xp = new AXIOMXPath("//wsp:Policy");
                xp.setNamespaceContext(nsCtx);
                OMElement policyEle = (OMElement) xp.selectSingleNode(builder.getDocumentElement());
                OMAttribute idAtt = policyEle.getAttribute(ID);
                if (idAtt == null) {
                    throw new AxisFault("The policy ID in the policy file " +
                                        policyFile.getAbsolutePath() + " is null");
                }
                if (serviceAdmin.removePolicy(serviceName,
                                              ServiceIdentifierDO.EMPTY_SERVICE_VERSION,
                                              idAtt.getAttributeValue())) {
                    break;
                }
            } catch (Exception e) {
                throw AxisFault.makeFault(e);
            }
        }
    }

    private void populateRampartConfig(RampartConfig rampartConfig,
                                       Properties props) throws AxisFault {
        if (rampartConfig != null) {

            if (!props.isEmpty()) {
                // Encryption crypto config
                {
                    CryptoConfig encrCryptoConfig = new CryptoConfig();
                    encrCryptoConfig.setProvider(ServerCrypto.class.getName());
                    encrCryptoConfig.setProp(props);
                    rampartConfig.setEncrCryptoConfig(encrCryptoConfig);
                }

                // Decryption crypto config
//            CryptoConfig decrCryptoConfig = new CryptoConfig();
//            decrCryptoConfig.setProvider(ServerCrypto.class.getName());
//            decrCryptoConfig.setProp(props);
//            rampartConfig.setDecCryptoConfig(decrCryptoConfig);

                // Signature crypto config
                {
                    CryptoConfig signatureCryptoConfig = new CryptoConfig();
                    signatureCryptoConfig.setProvider(ServerCrypto.class.getName());
                    signatureCryptoConfig.setProp(props);
                    rampartConfig.setSigCryptoConfig(signatureCryptoConfig);
                }
            }

            rampartConfig.setEncryptionUser(WSHandlerConstants.USE_REQ_SIG_CERT);
            rampartConfig.setUser(props.getProperty(USER));

            //Get ttl and timeskew params from axis2 xml
            int ttl = RampartConfig.DEFAULT_TIMESTAMP_TTL;
            int timeSkew = RampartConfig.DEFAULT_TIMESTAMP_MAX_SKEW;

            rampartConfig.setTimestampTTL(Integer.toString(ttl));
            rampartConfig.setTimestampMaxSkew(Integer.toString(timeSkew));
            rampartConfig.setTokenStoreClass(SecurityTokenStore.class.getName());
        }
    }

    /**
     * Assing UT related users and roles as well as keystore related trusted cert stores and private
     * key sote
     *
     * @param serviceName
     * @param serviceVersion
     * @param scenarioId
     * @param trustedCertStores
     * @param privateKeyStore
     * @param users
     * @param roles
     * @throws AxisFault TODO fix this for
     */
    public void assignUsersAndRolesAndKeyStores(String serviceName,
                                                String serviceVersion,
                                                String scenarioId,
                                                String[] trustedCertStores,
                                                String privateKeyStore,
                                                String[] users,
                                                String[] roles) throws AxisFault {
        //Assign users and roles
        //Add the callback handler instance
        AxisService axisService = getAxisConfig().getService(serviceName);
        if (axisService.getParameter(WSHandlerConstants.PW_CALLBACK_REF) == null) {
            axisService.addParameter(WSS4JUtil.getPasswordCallBackRefParameter(serviceName,
                                                                               serviceVersion));
        }

        if (serviceVersion == null || serviceVersion.trim().length() == 0) {
            serviceVersion = ServiceIdentifierDO.EMPTY_SERVICE_VERSION;
        }

        ServiceDO serviceDO = pm.getService(serviceName, serviceVersion);
        if (serviceDO == null) {
            throw new AxisFault("Service " + serviceName + " not found!");
        }

        for (Iterator iter = serviceDO.getUsers().iterator(); iter.hasNext();) {
            ServiceUserDO userDO = (ServiceUserDO) iter.next();
            userDO.getServices().remove(serviceDO);
            pm.updateEntity(userDO);
        }
        for (Iterator iter = serviceDO.getRoles().iterator(); iter.hasNext();) {
            ServiceUserRoleDO roleDO = (ServiceUserRoleDO) iter.next();
            roleDO.getServices().remove(serviceDO);
            pm.updateEntity(roleDO);
        }
        serviceDO.removeAllUsers();
        serviceDO.removeAllRoles();
        try {
            pm.updateService(serviceDO);
        } catch (ServiceNotFoundException e) {
            // Will not occur since it is already checked above
        }

        // assign users
        for (int i = 0; i < users.length; i++) {
            String user = users[i];
            if (user != null && user.trim().length() != 0) {
                serviceDO.setIsUTAuthEnabled(true);
                ServiceUserDO userDO = pm.getUser(user);
                serviceDO.addUser(userDO);
                pm.updateEntity(userDO);
            }
        }

        // assign roles
        for (int i = 0; i < roles.length; i++) {
            String role = roles[i];
            if (role != null && role.trim().length() != 0) {
                serviceDO.setIsUTAuthEnabled(true);
                ServiceUserRoleDO roleDO = pm.getRole(role);
                serviceDO.addRole(roleDO);
                pm.updateEntity(roleDO);
            }
        }

        try {
            pm.updateService(serviceDO);
        } catch (ServiceNotFoundException e) {
            // Will not occur since it is already checked above
        }

        //Assign key stores.
        serviceDO = pm.getService(serviceName, serviceVersion);
        pm.removeAllTrustedCertStores(serviceDO);
        serviceDO = pm.getService(serviceName, serviceVersion);
        /*serviceDO.setIsUTAuthEnabled(false);
        serviceDO.setIsExposedOnAllTransports(true);
        axisConfig.getService(serviceName).setEnableAllTransports(true);*/
        for (int i = 0; i < trustedCertStores.length; i++) {
            String ks = trustedCertStores[i];
            KeyStoreDO keyStore = pm.getKeyStore(ks);
            if (keyStore == null) {
                throw new AxisFault("Key Store " + ks + " does not exist");
            }
            serviceDO.addTrustedCertStore(keyStore);
        }
        KeyStoreDO pkKeyStore = pm.getKeyStore(privateKeyStore);
        serviceDO.setPrivateKeyStore(pkKeyStore);
        pm.updateEntity(serviceDO);

        configureScenario(serviceName,
                          scenarioId,
                          getServerCryptoProperties(trustedCertStores,
                                                    privateKeyStore,
                                                    pkKeyStore.getPrivateKeyAlias()));
        engageModules(serviceName, scenarioId);


    }

    public void assignKeyStores(String serviceName,
                                String scenarioId,
                                String[] trustedCertStores,
                                String privateKeyStore) throws AxisFault {

        ServiceDO serviceDO = pm.getService(serviceName,
                                            ServiceIdentifierDO.EMPTY_SERVICE_VERSION);
        pm.removeAllTrustedCertStores(serviceDO);
        serviceDO = pm.getService(serviceName,
                                  ServiceIdentifierDO.EMPTY_SERVICE_VERSION);
        serviceDO.setIsUTAuthEnabled(false);
        serviceDO.setIsExposedOnAllTransports(true);
        getAxisConfig().getService(serviceName).setEnableAllTransports(true);
        for (int i = 0; i < trustedCertStores.length; i++) {
            String ks = trustedCertStores[i];
            KeyStoreDO keyStore = pm.getKeyStore(ks);
            if (keyStore == null) {
                throw new AxisFault("Key Store " + ks + " does not exist");
            }
            serviceDO.addTrustedCertStore(keyStore);
        }
        KeyStoreDO pkKeyStore = pm.getKeyStore(privateKeyStore);
        serviceDO.setPrivateKeyStore(pkKeyStore);
        pm.updateEntity(serviceDO);

        configureScenario(serviceName,
                          scenarioId,
                          getServerCryptoProperties(trustedCertStores,
                                                    privateKeyStore,
                                                    pkKeyStore.getPrivateKeyAlias()));
        engageModules(serviceName, scenarioId);
    }

    public void assignUsersAndRoles(String serviceId,
                                    String serviceVersion,
                                    String scenarioId,
                                    String[] users,
                                    String[] roles) throws AxisFault {

        //Add the callback handler instance
        AxisService axisService = getAxisConfig().getService(serviceId);
        if (axisService.getParameter(WSHandlerConstants.PW_CALLBACK_REF) == null) {
            axisService.addParameter(WSS4JUtil.getPasswordCallBackRefParameter(
                    serviceId,
                    serviceVersion));
        }

        if (serviceVersion == null || serviceVersion.trim().length() == 0) {
            serviceVersion = ServiceIdentifierDO.EMPTY_SERVICE_VERSION;
        }

        ServiceDO serviceDO = pm.getService(serviceId, serviceVersion);
        if (serviceDO == null) {
            throw new AxisFault("Service " + serviceId + " not found!");
        }

        for (Iterator iter = serviceDO.getUsers().iterator(); iter.hasNext();) {
            ServiceUserDO userDO = (ServiceUserDO) iter.next();
            userDO.getServices().remove(serviceDO);
            pm.updateEntity(userDO);
        }
        for (Iterator iter = serviceDO.getRoles().iterator(); iter.hasNext();) {
            ServiceUserRoleDO roleDO = (ServiceUserRoleDO) iter.next();
            roleDO.getServices().remove(serviceDO);
            pm.updateEntity(roleDO);
        }
        serviceDO.removeAllUsers();
        serviceDO.removeAllRoles();
        try {
            pm.updateService(serviceDO);
        } catch (ServiceNotFoundException e) {
            // Will not occur since it is already checked above
        }

        // assign users
        for (int i = 0; i < users.length; i++) {
            String user = users[i];
            if (user != null && user.trim().length() != 0) {
                serviceDO.setIsUTAuthEnabled(true);
                ServiceUserDO userDO = pm.getUser(user);
                serviceDO.addUser(userDO);
                pm.updateEntity(userDO);
            }
        }

        // assign roles
        for (int i = 0; i < roles.length; i++) {
            String role = roles[i];
            if (role != null && role.trim().length() != 0) {
                serviceDO.setIsUTAuthEnabled(true);
                ServiceUserRoleDO roleDO = pm.getRole(role);
                serviceDO.addRole(roleDO);
                pm.updateEntity(roleDO);
            }
        }

        try {
            pm.updateService(serviceDO);
        } catch (ServiceNotFoundException e) {
            // Will not occur since it is already checked above
        }

        configureScenario(serviceId, scenarioId, new Properties());
        engageModules(serviceId, scenarioId);
    }

    private void engageModules(String serviceId,
                               String scenarioId) throws AxisFault {
        SecurityScenarioDO securityScenario = pm.getSecurityScenario(scenarioId);
        AxisService axisService = getAxisConfig().getService(serviceId);
        for (Iterator modules = securityScenario.modules.iterator(); modules.hasNext();) {
            ModuleDO moduleDO = (ModuleDO) modules.next();
            String moduleName = moduleDO.getModuleIdentifierDO().getName();
            if (!axisService.isEngaged(moduleName)) {
                AxisModule axisModule = getAxisConfig().getModule(moduleName);
                new ModuleAdmin().
                        engageModuleForService(axisModule.getName(),
                                               axisModule.getVersion(),
                                               serviceId,
                                               ServiceIdentifierDO.EMPTY_SERVICE_VERSION);
            }
        }
    }

    private void disengageModules(String serviceId,
                                  String scenarioId) throws AxisFault {
        SecurityScenarioDO securityScenario = pm.getSecurityScenario(scenarioId);
        AxisConfiguration axisConfig = getAxisConfig();
        AxisService axisService = axisConfig.getService(serviceId);
        ModuleAdmin moduleAdmin = new ModuleAdmin();
        for (Iterator modules = securityScenario.modules.iterator(); modules.hasNext();) {
            ModuleDO moduleDO = (ModuleDO) modules.next();
            String moduleName = moduleDO.getModuleIdentifierDO().getName();
            String moduleVersion = moduleDO.getModuleIdentifierDO().getVersion();
            AxisModule axisModule = axisConfig.getModule(moduleName, moduleVersion);
            if (axisService.isEngaged(axisModule)) {
                moduleAdmin.forceDisengageModuleFromService(axisModule.getName(),
                                                            axisModule.getVersion(),
                                                            serviceId,
                                                            ServiceIdentifierDO.EMPTY_SERVICE_VERSION);
            }
        }
    }

    public SecurityAssignment getSecurityAssignment(String serviceId,
                                                    String serviceVersion) throws AxisFault {

        SecurityAssignment assignment = new SecurityAssignment();

        if (serviceVersion == null || serviceVersion.trim().length() == 0) {
            serviceVersion = ServiceIdentifierDO.EMPTY_SERVICE_VERSION;
        }
        ServiceDO serviceDO = pm.getService(serviceId, serviceVersion);
        if (serviceDO == null) {
            throw new AxisFault("Service " + serviceId + " not found!");
        }

        // User Assignments
        {
            List list = new ArrayList();
            ServiceUserDO[] users = pm.getUsers(); // All users
            for (int i = 0; i < users.length; i++) {
                SelectableString user = new SelectableString();
                ServiceUserDO userDO = users[i];
                user.setString(userDO.getUsername());
                for (Iterator iterator =
                        serviceDO.getUsers().iterator(); iterator.hasNext();) {
                    ServiceUserDO selectedUser = (ServiceUserDO) iterator.next();
                    if (userDO.getUsername().equals(selectedUser.getUsername())) {
                        user.setSelected(true);
                    }
                }
                list.add(user);
            }
            assignment.setUsers((SelectableString[]) list.toArray(new SelectableString[list.size()]));
        }

        // Role Assignments
        {
            List list = new ArrayList();
            ServiceUserRoleDO[] roles = pm.getUserRoles(); // All roles
            for (int i = 0; i < roles.length; i++) {
                SelectableString role = new SelectableString();
                ServiceUserRoleDO roleDO = roles[i];
                role.setString(roleDO.getRole());
                for (Iterator iterator =
                        serviceDO.getRoles().iterator(); iterator.hasNext();) {
                    ServiceUserRoleDO selectedRole = (ServiceUserRoleDO) iterator.next();
                    if (roleDO.getRole().equals(selectedRole.getRole())) {
                        role.setSelected(true);
                    }
                }
                list.add(role);
            }
            assignment.setRoles((SelectableString[]) list.toArray(new SelectableString[list.size()]));
        }
        return assignment;
    }

    /**
     * Infomation related to keystore assignment.
     *
     * @param serviceName
     * @return UtKeystoreSecurityAssignment
     * @throws AxisFault
     */
    public UtKeystoreSecurityAssignment getUtKeystoreSecurityAssignment(String serviceName)
            throws AxisFault {
        UtKeystoreSecurityAssignment utKeystore = new UtKeystoreSecurityAssignment();
        utKeystore.setUt(getSecurityAssignment(serviceName, null));
        CryptoAdmin ca = new CryptoAdmin();
        utKeystore.setKeystore(ca.getServiceKeyStores(serviceName));
        return utKeystore;
    }

    private Properties getServerCryptoProperties(String[] trustedCertStores,
                                                 String privateKeyStore,
                                                 String privKeyAlias) {

        Properties props = RampartConfigUtil.getServerCryptoProperties(
                trustedCertStores, privateKeyStore, privKeyAlias);
        props.setProperty(USER, privKeyAlias);
        return props;
    }
}
