/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.admin.service;

import org.apache.axis2.AxisFault;
import org.apache.axis2.context.ConfigurationContext;
import org.apache.sandesha2.client.SandeshaClient;
import org.apache.sandesha2.client.SandeshaReport;
import org.apache.sandesha2.client.SequenceReport;
import org.wso2.utils.AbstractAdmin;
import org.wso2.wsas.admin.service.util.Sandesha2Data;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * Admin Service to manage Sandesha2.
 */
public class Sandesha2Admin extends AbstractAdmin {

    /**
     * Gives all Outgoing sequences. An Outgoing Sequence is a sequence to which this endpoint works as a RMS.
     *
     * @return A array of Strings. Each entry gives the SequenceID of an Outgoing Sequence.
     * @throws AxisFault
     */
    public String[] getOutgoingSequences() throws AxisFault {
        SandeshaReport sandeshaReport = getSandeshaReport();
        List outgoingSequnces = sandeshaReport.getOutgoingSequenceList();
        return (String[]) outgoingSequnces.toArray(new String[outgoingSequnces.size()]);

    }

    /**
     * convenience method to return observation data
     *
     * @return Sandesha2Data array
     * @throws AxisFault
     */
    public Sandesha2Data[] getOutgoingSequenceData() throws AxisFault {

        SandeshaReport sandeshaReport = getSandeshaReport();
        List outgoingSequnces = sandeshaReport.getOutgoingSequenceList();
        ArrayList dataList = new ArrayList();
        for (Iterator iterator = outgoingSequnces.iterator(); iterator.hasNext();) {
            String sequenceId = (String) iterator.next();
            Sandesha2Data s2Data = new Sandesha2Data();
            s2Data.setSequenceID(sequenceId);
            s2Data.setFlow(2);
            s2Data.setSequenceReport(getOutgoingSequenceDetails(sequenceId));
            dataList.add(s2Data);
        }

        sandeshaReport = getSandeshaReport();
        List incomingSequences = sandeshaReport.getIncomingSequenceList();
        for (Iterator iterator = incomingSequences.iterator(); iterator.hasNext();) {
            String sequenceId = (String) iterator.next();
            Sandesha2Data s2Data = new Sandesha2Data();
            s2Data.setSequenceID(sequenceId);
            s2Data.setFlow(1);
            s2Data.setSequenceReport(getIncomingSequenceDetails(sequenceId));
            dataList.add(s2Data);
        }

        Sandesha2Data[] arr = new Sandesha2Data[dataList.size()];
        for (int i = 0; i < arr.length; i++) {
            Sandesha2Data data = (Sandesha2Data) dataList.get(i);
            arr[i] = data;
        }

        return arr;
    }

    /**
     * Gives all Incoming sequences. An incoming Sequence is a sequence to which this endpoint works as a RMD.
     *
     * @return A list of Strings. Each entry gives the SequenceID of an Incoming Sequence.
     * @throws AxisFault
     */
    public String[] getIncomingSequences() throws AxisFault {
        SandeshaReport sandeshaReport = getSandeshaReport();
        List incomingSequences = sandeshaReport.getOutgoingSequenceList();
        return (String[]) incomingSequences.toArray(new String[incomingSequences.size()]);
    }

    /**
     * Gives a report for an Incoming Sequence.
     *
     * @param sequenceID The ID of the Sequence
     * @return SequenceReport
     * @throws AxisFault
     */
    public SequenceReport getIncomingSequenceDetails(String sequenceID) throws AxisFault {
        ConfigurationContext configurationContext = getConfigContext();
        if (configurationContext == null) {
            throw new AxisFault("Cannot proceed. ConfigurationContext is not set");
        }

        SequenceReport incomingSequenceReport =
                SandeshaClient.getIncomingSequenceReport(sequenceID, configurationContext);
        return incomingSequenceReport;
    }

    /**
     * Gives an report for an Outgoing Sequence.
     *
     * @param sequenceID The ID of the Sequence.
     * @return SequenceReport
     * @throws AxisFault
     */
    public SequenceReport getOutgoingSequenceDetails(String sequenceID) throws AxisFault {
        ConfigurationContext configurationContext = getConfigContext();
        if (configurationContext == null) {
            throw new AxisFault("Cannot proceed. ConfigurationContext is not set");
        }

        SequenceReport outgoingSequenceReport =
                SandeshaClient.getOutgoingSequenceReport(sequenceID, configurationContext);
        return outgoingSequenceReport;
    }

    private SandeshaReport getSandeshaReport() throws AxisFault {
        ConfigurationContext configurationContext = getConfigContext();
        if (configurationContext == null) {
            throw new AxisFault("Cannot proceed. ConfigurationContext is not set");
        }

        SandeshaReport sandeshaReport = SandeshaClient.getSandeshaReport(configurationContext);
        if (sandeshaReport == null) {
            throw new AxisFault("Could not find a valid SandeshaReport");
        }

        return sandeshaReport;
    }

}
