/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.admin.service;

import org.apache.axiom.om.OMElement;
import org.apache.axis2.AxisFault;
import org.apache.axis2.Constants;
import org.apache.axis2.description.AxisOperation;
import org.apache.axis2.description.AxisService;
import org.apache.axis2.description.AxisServiceGroup;
import org.apache.axis2.description.Parameter;
import org.apache.axis2.description.PolicyInclude;
import org.apache.axis2.engine.Handler;
import org.apache.axis2.engine.Phase;
import org.apache.axis2.phaseresolver.PhaseMetadata;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.neethi.Policy;
import org.wso2.utils.AbstractAdmin;
import org.wso2.wsas.admin.service.util.OperationMetaData;
import org.wso2.wsas.persistence.PersistenceManager;
import org.wso2.wsas.persistence.dataobject.OperationDO;
import org.wso2.wsas.persistence.dataobject.OperationParameterDO;
import org.wso2.wsas.persistence.dataobject.ServiceIdentifierDO;
import org.wso2.wsas.persistence.exception.DuplicateEntityException;
import org.wso2.wsas.util.ParameterUtil;
import org.wso2.wsas.util.PolicyUtil;

import javax.xml.namespace.QName;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;


/**
 * Admin service to manage service oprations
 */
public class OperationAdmin extends AbstractAdmin {
    private static final Log log = LogFactory.getLog(OperationAdmin.class);

    /**
     * list all the operations in the service including both control and published operations
     *
     * @param serviceName
     * @return list of all operations
     * @throws AxisFault
     */
    public OperationMetaData[] listAllOperations(String serviceName) throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceName);
        List opList = new ArrayList();

        for (Iterator operations = axisService.getOperations(); operations.hasNext();) {
            AxisOperation axisOperation = (AxisOperation) operations.next();
            if (axisOperation.getName() != null) {
                opList.add(getOperationMetaData(serviceName,
                                                axisOperation.getName().getLocalPart()));
            }
        }
        Collections.sort(opList, new Comparator() {
            public int compare(Object arg0, Object arg1) {
                OperationMetaData a = (OperationMetaData) arg0;
                OperationMetaData b = (OperationMetaData) arg1;
                return a.getName().compareToIgnoreCase(b.getName());
            }
        });
        return (OperationMetaData[]) opList.toArray(new OperationMetaData[opList.size()]);
    }

    /**
     * list only the control operations added by modules and some other way
     *
     * @param serviceName
     * @return list of control operations
     * @throws AxisFault
     */
    public OperationMetaData[] listControlOperations(String serviceName) throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceName);
        List opMetaDataList = new ArrayList();
        for (Iterator operations = axisService.getControlOperations().iterator();
             operations.hasNext();) {
            AxisOperation axisOperation = (AxisOperation) operations.next();
            opMetaDataList.add(getOperationMetaData(serviceName,
                                                    axisOperation.getName().getLocalPart()));
        }
        return (OperationMetaData[]) opMetaDataList.
                toArray(new OperationMetaData[opMetaDataList.size()]);
    }

    /**
     * list all the published operations (come from servics.xml and wsdl)
     *
     * @param serviceName
     * @return list of published operations
     * @throws AxisFault
     */
    public OperationMetaData[] listPublishedOperations(String serviceName) throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceName);
        List opList = new ArrayList();

        for (Iterator operations = axisService.getPublishedOperations().iterator();
             operations.hasNext();) {
            AxisOperation axisOperation = (AxisOperation) operations.next();
            opList.add(getOperationMetaData(serviceName,
                                            axisOperation.getName().getLocalPart()));
        }

        return (OperationMetaData[]) opList.toArray(new OperationMetaData[opList.size()]);
    }

    /**
     * return all available operation metadata (not counts)
     *
     * @param serviceId
     * @param serviceVersion
     * @param operationId
     * @return operation info
     * @throws AxisFault
     */
    public Object[] getOperationInfo(String serviceId,
                                     String serviceVersion,
                                     String operationId) throws AxisFault {
        throw new AxisFault("Operation not yet implemented");
    }

    /**
     * return all accumulated data about this operation
     *
     * @param serviceName
     * @param operationName
     * @return operation stats
     * @throws AxisFault
     */
    public OperationMetaData getOperationStatistics(String serviceName,
                                                    String operationName) throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceName);
        if (axisService != null) {
            return getOperationMetaData(serviceName, operationName);
        }
        return null;
    }

    /**
     * Configuring MTOM
     *
     * @param flag
     * @param serviceName
     * @param operationName
     * @throws AxisFault
     */
    public void configureMTOM(String flag, String serviceName, String operationName)
            throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceName);
        AxisOperation operation;
        if (axisService != null) {
            operation = axisService.getOperation(new QName(operationName));
        } else {
            throw new AxisFault("Service cannot be found");
        }
        if (operation == null) {
            throw new AxisFault("Operation cannot be found");
        }

        ArrayList parameters = operation.getParameters();

        boolean found = false;
        for (Iterator iterator = parameters.iterator(); iterator.hasNext();) {
            Parameter parameter = (Parameter) iterator.next();

            if (parameter.getParameterType() == Parameter.TEXT_PARAMETER &&
                parameter.getValue().toString().equals(Constants.Configuration.ENABLE_MTOM)) {
                parameter.setValue(flag.trim());
                found = true;
                break;
            }

        }
        if (!found) {
            Parameter parameter = ParameterUtil
                    .createParameter(Constants.Configuration.ENABLE_MTOM, flag.trim());
            operation.addParameter(parameter);
        }

        Parameter parameter = operation.getParameter(Constants.Configuration.ENABLE_MTOM);
        //At this point parameter will not be null;

        //Persisting the parameter
        PersistenceManager pm = new PersistenceManager();

        OperationParameterDO paramDO =
                pm.getOperationParameter(serviceName,
                                         ServiceIdentifierDO.EMPTY_SERVICE_VERSION,
                                         operationName,
                                         parameter.getName());
        if (paramDO != null) {
            paramDO.setValue(parameter.getParameterElement().toString());
            pm.updateEntity(paramDO);
        } else {
            paramDO = new OperationParameterDO();
            paramDO.setName(parameter.getName());
            paramDO.setValue(parameter.getParameterElement().toString());
            OperationDO opDO = pm.getOperation(serviceName,
                                               ServiceIdentifierDO.EMPTY_SERVICE_VERSION,
                                               operationName);
            paramDO.setOperation(opDO);

            try {
                pm.addEntity(paramDO);
            } catch (DuplicateEntityException e) {
                log.error("Operation Parameter already exists", e);
            }

        }

    }

    /**
     * return all parameters for this operation (including inherited ones),
     * where each parameter is an XML fragment representing the "parameter" element
     *
     * @param serviceId
     * @param operationId
     * @return operation params
     * @throws AxisFault
     */
    public OMElement[] getOperationParameters(String serviceId,
                                              String operationId) throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceId);
        AxisOperation op = axisService.getOperation(new QName(operationId));

        if (op == null) {
            throw new AxisFault("Invalid operation : " + operationId +
                                " not available in service : " + serviceId);
        }

        ArrayList allParameter = new ArrayList();
        ArrayList globalParameters = getAxisConfig().getParameters();

        for (int i = 0; i < globalParameters.size(); i++) {
            Parameter parameter = (Parameter) globalParameters.get(i);
            allParameter.add(parameter.getParameterElement());
        }

        AxisService service = getAxisConfig().getService(serviceId);

        if (service == null) {
            throw new AxisFault("invalid service name");
        }

        ArrayList serviceParams = service.getParameters();

        for (int i = 0; i < serviceParams.size(); i++) {
            Parameter parameter = (Parameter) serviceParams.get(i);
            allParameter.add(parameter.getParameterElement());
        }

        AxisServiceGroup axisServiceGroup = (AxisServiceGroup) service.getParent();
        ArrayList serviceGroupParams = axisServiceGroup.getParameters();

        for (int i = 0; i < serviceGroupParams.size(); i++) {
            Parameter parameter = (Parameter) serviceGroupParams.get(i);
            allParameter.add(parameter.getParameterElement());
        }

        ArrayList opParams = op.getParameters();

        for (int i = 0; i < opParams.size(); i++) {
            Parameter parameter = (Parameter) opParams.get(i);
            allParameter.add(parameter.getParameterElement());
        }

        return (OMElement[]) allParameter.toArray(new OMElement[allParameter.size()]);
    }

    /**
     * return only the parameters for explicitly set for this operation
     * (not including inherited ones), where each parameter is an XML fragment
     * representing the "parameter" element
     *
     * @param serviceName
     * @param operationName
     * @return declared operation params
     * @throws AxisFault
     */
    public OMElement[] getDeclaredOperationParameters(String serviceName,
                                                      String operationName) throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceName);
        AxisOperation op = axisService.getOperation(new QName(operationName));

        if (op == null) {
            throw new AxisFault("Invalid operation : " + operationName +
                                " not available in service : " + serviceName);
        }

        ArrayList allParameter = new ArrayList();
        ArrayList opParams = op.getParameters();

        for (int i = 0; i < opParams.size(); i++) {
            Parameter parameter = (Parameter) opParams.get(i);
            OMElement element = parameter.getParameterElement();
            if (element != null) {
                allParameter.add(element);
            }
        }

        return (OMElement[]) allParameter.toArray(new OMElement[allParameter.size()]);
    }

    public void setOperationParameters(String serviceId,
                                       String operationId,
                                       String groupVersion,
                                       OMElement[] parameterElements) throws AxisFault {
        for (int i = 0; i < parameterElements.length; i++) {
            setOperationParameter(serviceId, operationId, groupVersion, parameterElements[i]);
        }

    }

    public void removeOperationParameters(String serviceId,
                                          String operationId,
                                          String groupVersion,
                                          OMElement[] parameterElements) throws AxisFault {
        for (int i = 0; i < parameterElements.length; i++) {
            removeOperationParameter(serviceId, operationId, groupVersion, parameterElements[i]);
        }

    }

    public void setOperationParameter(String serviceId,
                                      String operationId,
                                      String groupVersion,
                                      OMElement parameterElement) throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceId);

        if (axisService == null) {
            throw new AxisFault("invalid service name service not found : " +
                                serviceId);
        }

        AxisOperation axisOp = axisService.getOperation(new QName(operationId));

        if (axisOp == null) {
            throw new AxisFault("Invalid operation : " + operationId +
                                " not available in service : " + serviceId);
        }

        Parameter parameter = ParameterUtil.createParameter(parameterElement);

        // Persist the parameter
        PersistenceManager pm = new PersistenceManager();
        Parameter p = axisOp.getParameter(parameter.getName());
        OperationParameterDO paramDO =
                pm.getOperationParameter(serviceId,
                                         ServiceIdentifierDO.EMPTY_SERVICE_VERSION,
                                         operationId,
                                         parameter.getName());
        if (p != null && paramDO != null) {
            // Setting a new value for a parameter declared in AxisOperation
            if (!p.isLocked()) {
                axisOp.addParameter(parameter);
                paramDO.setValue(parameterElement.toString());
                pm.updateEntity(paramDO);
            }
        } else {
            // If we are adding a new AxisOperation param or overriding a param in the Configuration hierarchy

            if (p == null || !p.isLocked()) {
                axisOp.addParameter(parameter);

                paramDO = new OperationParameterDO();
                paramDO.setName(parameter.getName());
                paramDO.setValue(parameterElement.toString());

                OperationDO opDO = pm.getOperation(serviceId,
                                                   ServiceIdentifierDO.EMPTY_SERVICE_VERSION,
                                                   operationId);
                paramDO.setOperation(opDO);

                try {
                    pm.addEntity(paramDO);
                } catch (DuplicateEntityException e) {
                    log.error("Operation Parameter already exists", e);
                }
            }
        }
    }

    public void removeOperationParameter(String serviceId,
                                         String operationId,
                                         String groupVersion,
                                         OMElement parameterElement) throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceId);

        if (axisService == null) {
            throw new AxisFault("invalid service name service not found : " +
                                serviceId);
        }

        AxisOperation axisOp = axisService.getOperation(new QName(operationId));

        if (axisOp == null) {
            throw new AxisFault("Invalid operation : " + operationId +
                                " not available in service : " + serviceId);
        }

        Parameter parameter = ParameterUtil.createParameter(parameterElement);

        // Un-Persist the parameter
        PersistenceManager pm = new PersistenceManager();
        Parameter p = axisOp.getParameter(parameter.getName());
        OperationParameterDO paramDO =
                pm.getOperationParameter(serviceId,
                                         ServiceIdentifierDO.EMPTY_SERVICE_VERSION,
                                         operationId,
                                         parameter.getName());
        if (p != null && paramDO != null) {
            // Setting a new value for a parameter declared in AxisOperation
            if (!p.isLocked()) {
                axisOp.removeParameter(parameter);
                pm.deleteEntity(paramDO);
            }
        } else {

            if (p == null || !p.isLocked()) {
                axisOp.removeParameter(parameter);
            }
        }
    }

    /**
     * list all the operation phases
     *
     * @param serviceName
     * @param operationName
     * @param flow
     * @return operation phases
     * @throws AxisFault
     */
    public String[] listOperationPhases(String serviceName,
                                        String operationName,
                                        int flow) throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceName);
        AxisOperation op = axisService.getOperation(new QName(operationName));

        if (op == null) {
            throw new AxisFault("Invalid operation : " + operationName +
                                " not available in service : " + serviceName);
        }

        String[] phaseNames = null;

        switch (flow) {
            case PhaseMetadata.IN_FLOW: {
                ArrayList inflow = op.getRemainingPhasesInFlow();
                phaseNames = new String[inflow.size()];

                for (int i = 0; i < inflow.size(); i++) {
                    Phase phase = (Phase) inflow.get(i);
                    phaseNames[i] = phase.getPhaseName();
                }

                break;
            }

            case PhaseMetadata.OUT_FLOW: {
                ArrayList inflow = op.getPhasesOutFlow();
                phaseNames = new String[inflow.size()];

                for (int i = 0; i < inflow.size(); i++) {
                    Phase phase = (Phase) inflow.get(i);
                    phaseNames[i] = phase.getPhaseName();
                }

                break;
            }

            case PhaseMetadata.FAULT_IN_FLOW: {
                ArrayList inflow = op.getPhasesInFaultFlow();
                phaseNames = new String[inflow.size()];

                for (int i = 0; i < inflow.size(); i++) {
                    Phase phase = (Phase) inflow.get(i);
                    phaseNames[i] = phase.getPhaseName();
                }

                break;
            }

            case PhaseMetadata.FAULT_OUT_FLOW: {
                ArrayList inflow = op.getPhasesOutFaultFlow();
                phaseNames = new String[inflow.size()];

                for (int i = 0; i < inflow.size(); i++) {
                    Phase phase = (Phase) inflow.get(i);
                    phaseNames[i] = phase.getPhaseName();
                }

                break;
            }
        }

        return phaseNames;
    }

    /**
     * To list handlers in a given operation phases
     *
     * @param serviceName
     * @param operationName
     * @param flow
     * @param phaseName
     * @return operation phase handlers
     * @throws AxisFault
     */
    public String[] listOperationPhaseHandlers(String serviceName,
                                               String operationName,
                                               int flow,
                                               String phaseName) throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceName);
        AxisOperation op = axisService.getOperation(new QName(operationName));

        if (op == null) {
            throw new AxisFault("Invalid operation : " + operationName +
                                " not available in service : " + serviceName);
        }

        String[] handlers = null;

        switch (flow) {
            case PhaseMetadata.IN_FLOW: {
                ArrayList inflow = op.getRemainingPhasesInFlow();

                for (int i = 0; i < inflow.size(); i++) {
                    Phase phase = (Phase) inflow.get(i);

                    if (phase.getPhaseName().equals(phaseName)) {
                        handlers = new String[phase.getHandlerCount()];

                        ArrayList hands = phase.getHandlers();

                        for (int j = 0; j < hands.size(); j++) {
                            Handler handler = (Handler) hands.get(j);
                            handlers[j] = handler.getName();
                        }
                    }
                }

                break;
            }

            case PhaseMetadata.OUT_FLOW: {
                ArrayList inflow = op.getPhasesOutFlow();

                for (int i = 0; i < inflow.size(); i++) {
                    Phase phase = (Phase) inflow.get(i);

                    if (phase.getPhaseName().equals(phaseName)) {
                        handlers = new String[phase.getHandlerCount()];

                        ArrayList hands = phase.getHandlers();

                        for (int j = 0; j < hands.size(); j++) {
                            Handler handler = (Handler) hands.get(j);
                            handlers[j] = handler.getName();
                        }
                    }
                }

                break;
            }

            case PhaseMetadata.FAULT_IN_FLOW: {
                ArrayList inflow = op.getPhasesInFaultFlow();

                for (int i = 0; i < inflow.size(); i++) {
                    Phase phase = (Phase) inflow.get(i);

                    if (phase.getPhaseName().equals(phaseName)) {
                        handlers = new String[phase.getHandlerCount()];

                        ArrayList hands = phase.getHandlers();

                        for (int j = 0; j < hands.size(); j++) {
                            Handler handler = (Handler) hands.get(j);
                            handlers[j] = handler.getName();
                        }
                    }
                }

                break;
            }

            case PhaseMetadata.FAULT_OUT_FLOW: {
                ArrayList inflow = op.getPhasesOutFaultFlow();

                for (int i = 0; i < inflow.size(); i++) {
                    Phase phase = (Phase) inflow.get(i);

                    if (phase.getPhaseName().equals(phaseName)) {
                        handlers = new String[phase.getHandlerCount()];

                        ArrayList hands = phase.getHandlers();

                        for (int j = 0; j < hands.size(); j++) {
                            Handler handler = (Handler) hands.get(j);
                            handlers[j] = handler.getName();
                        }
                    }
                }

                break;
            }
        }

        return handlers;
    }

    public OMElement getPolicy(String serviceId, String operationId) throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceId);

        if (axisService == null) {
            throw new AxisFault("Invalid service : " + serviceId);
        }

        AxisOperation axisOperation = axisService.getOperation(new QName(
                operationId));

        if (axisOperation == null) {
            throw new AxisFault("Invalid operation : " + operationId +
                                " not available in service : " + serviceId);
        }

        PolicyInclude operationPolicyInclude = axisOperation.getPolicyInclude();
        Policy operationPolicy = operationPolicyInclude.getPolicy();

        if (operationPolicy == null) {
            return PolicyUtil.getEmptyPolicyAsOMElement();
        }

        return PolicyUtil.getPolicyAsOMElement(operationPolicy);
    }

    public void setPolicy(String serviceId,
                          String operationId,
                          OMElement policyElement) throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceId);

        if (axisService == null) {
            throw new AxisFault("Invalid service : " + serviceId);
        }
        AxisOperation axisOperation = axisService.getOperation(new QName(operationId));
        if (axisOperation == null) {
            throw new AxisFault("Invalid operation : " + operationId +
                                " not available in service : " + serviceId);
        }
        Policy operationPolicy;
        try {
            operationPolicy = PolicyUtil.getPolicyFromOMElement(policyElement);
        } catch (RuntimeException ex) {
            throw new RuntimeException("Cannot convert the OMElement to Policy",
                                       ex);
        }
        PolicyInclude operationPolicyInclude = axisOperation.getPolicyInclude();
        operationPolicyInclude.setPolicy(operationPolicy);
    }

    private OperationMetaData getOperationMetaData(String serviceName,
                                                   String operationName) throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceName);
        AxisOperation axisOperation = null;
        if (axisService != null) {
            axisOperation = axisService.getOperation(new QName(operationName));
        }
        if (axisOperation == null) {
            return null;
        }
        OperationMetaData opMetaData = new OperationMetaData();
        opMetaData.setName(axisOperation.getName().getLocalPart());
        opMetaData.setControlOperation(axisOperation.isControlOperation());
        Parameter parameter = axisOperation.getParameter(Constants.Configuration.ENABLE_MTOM);
        if (parameter != null) {
            opMetaData.setEnableMTOM((String) parameter.getValue());
        } else {
            opMetaData.setEnableMTOM("false");
        }

        try {
            StatisticsClient statClient = new StatisticsClient();

            opMetaData.setOpRequestCount(statClient.getOperationRequestCount(serviceName,
                                                                             operationName));
            opMetaData.setOpResponseCount(statClient.getOperationResponseCount(serviceName,
                                                                               operationName));
            opMetaData.setOpFaultCount(statClient.getOperationFaultCount(serviceName,
                                                                         operationName));
            opMetaData.setMaxResponseTime(statClient.getMaxOperationResponseTime(serviceName,
                                                                                 operationName));
            opMetaData.setMinResponseTime(statClient.getMinOperationResponseTime(serviceName,
                                                                                 operationName));
            opMetaData.setAvgResponseTime(statClient.getAvgOperationResponseTime(serviceName,
                                                                                 operationName));
        } catch (Exception e) {
            throw AxisFault.makeFault(e);
        }
        return opMetaData;
    }
}
