/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.admin.service;

import org.apache.axiom.om.OMAttribute;
import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.apache.axis2.AxisFault;
import org.apache.axis2.description.AxisModule;
import org.apache.axis2.description.AxisOperation;
import org.apache.axis2.description.AxisService;
import org.apache.axis2.description.AxisServiceGroup;
import org.apache.axis2.description.Parameter;
import org.apache.axis2.description.PolicyInclude;
import org.apache.axis2.engine.AxisConfiguration;
import org.apache.axis2.util.Utils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.neethi.Constants;
import org.apache.neethi.Policy;
import org.apache.neethi.PolicyRegistry;
import org.apache.rahas.impl.SCTIssuerConfig;
import org.apache.rahas.impl.TokenCancelerConfig;
import org.wso2.utils.AbstractAdmin;
import org.wso2.utils.FileManipulator;
import org.wso2.utils.ServerConfiguration;
import org.wso2.utils.ServerException;
import org.wso2.wsas.ServerConstants;
import org.wso2.wsas.admin.service.util.ModuleMetaData;
import org.wso2.wsas.admin.service.util.PolicyData;
import org.wso2.wsas.persistence.PersistenceManager;
import org.wso2.wsas.persistence.dataobject.KeyStoreDO;
import org.wso2.wsas.persistence.dataobject.ModuleDO;
import org.wso2.wsas.persistence.dataobject.ModuleIdentifierDO;
import org.wso2.wsas.persistence.dataobject.ModuleParameterDO;
import org.wso2.wsas.persistence.dataobject.ModulePolicyDO;
import org.wso2.wsas.persistence.dataobject.SecurityScenarioDO;
import org.wso2.wsas.persistence.dataobject.ServiceDO;
import org.wso2.wsas.persistence.dataobject.ServiceIdentifierDO;
import org.wso2.wsas.persistence.exception.DuplicateEntityException;
import org.wso2.wsas.persistence.exception.ModuleNotFoundException;
import org.wso2.wsas.persistence.exception.ModulePolicyNotFoundException;
import org.wso2.wsas.security.RahasUtil;
import org.wso2.wsas.security.ServerCrypto;
import org.wso2.wsas.util.ParameterUtil;
import org.wso2.wsas.util.PolicyUtil;
import org.wso2.wsas.util.SystemFilter;
import org.wso2.wsas.util.WsasUtils;

import javax.xml.namespace.QName;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;


/**
 * Admin service to manage operations related to modules
 */
public class ModuleAdmin extends AbstractAdmin {


    private static final String RAHAS = "rahas";
    private static final String RAMPART = "rampart";
    private static Log log = LogFactory.getLog(ModuleAdmin.class);

    private PersistenceManager pm = new PersistenceManager();

    public ModuleAdmin() {
    }


    /**
     * List all the available modules
     *
     * @return list of modules as a ModuleMetadata[].
     * @throws AxisFault
     */
    public ModuleMetaData[] listModules() throws AxisFault {
        ArrayList moduleList = new ArrayList();

        for (Iterator moduleIter = getAxisConfig().getModules().values().iterator();
             moduleIter.hasNext();) {
            AxisModule axisModule = (AxisModule) moduleIter.next();

            if (SystemFilter.isFilteredOutModule(axisModule.getName())) {
                continue;
            }

            ModuleMetaData md = new ModuleMetaData();
            md.setModulename(axisModule.getName());
            md.setModuleId(axisModule.getName());
            md.setEngaged("" + getAxisConfig().isEngaged(axisModule));

            String description = axisModule.getModuleDescription();

            if (description != null) {
                md.setDescription(description);
            } else {
                md.setDescription("No description found");
            }

            String moduleVersion = axisModule.getVersion();
            if (moduleVersion != null) {
                md.setModuleVersion(moduleVersion);
            } else {
                md.setModuleVersion("");
            }

            moduleList.add(md);
        }

        return (ModuleMetaData[]) moduleList.toArray(new ModuleMetaData[moduleList.size()]);
    }

    /**
     * List all the globally engaged modules
     *
     * @return list of globally engaged modules as a String[].
     * @throws AxisFault
     */
    public ModuleMetaData[] listGloballyEngagesModules() throws AxisFault {
        ArrayList emodules = new ArrayList();

        for (Iterator engagedModules = getAxisConfig().getEngagedModules().iterator();
             engagedModules.hasNext();) {
            AxisModule axisModule = (AxisModule) engagedModules.next();
            String name = axisModule.getName();

            if (!SystemFilter.isFilteredOutModule(name)) {
                emodules.add(new ModuleMetaData(name, axisModule.getVersion()));
            }
        }
        return (ModuleMetaData[]) emodules.toArray(new ModuleMetaData[emodules.size()]);
    }

    /**
     * List modules that are engaged to the service name
     *
     * @param serviceName
     * @return list of service modules as a String[].
     * @throws AxisFault
     */
    public ModuleMetaData[] listServiceModules(String serviceName) throws AxisFault {
        AxisService service = getAxisConfig().getService(serviceName);
        if (service == null) {
            throw new AxisFault("Service not found or invalid service - " + serviceName);
        }
        List engageModules = new ArrayList();
        for (Iterator engagedModulesIter = service.getEngagedModules().iterator();
             engagedModulesIter.hasNext();) {
            AxisModule module = (AxisModule) engagedModulesIter.next();

            if (!(getAxisConfig().isEngaged(module) || service.getParent().isEngaged(module))) {
                if (!SystemFilter.isFilteredOutModule(module.getName())) {
                    engageModules.add(new ModuleMetaData(module.getName(), module.getVersion()));
                }
            }
        }
        return (ModuleMetaData[]) engageModules.toArray(new ModuleMetaData[engageModules.size()]);
    }

    /**
     * List modules that are engaged to the service group
     *
     * @param serviceGroupId
     * @return list of service group modules as a String[].
     * @throws AxisFault
     */
    public ModuleMetaData[] listServiceGroupModules(String serviceGroupId) throws AxisFault {
        AxisServiceGroup asg = getAxisConfig().getServiceGroup(serviceGroupId);
        if (asg == null) {
            throw new AxisFault(
                    "Service Group " + serviceGroupId + " not found or invalid service group");
        }
        List engagedModules = new ArrayList();
        for (Iterator iterator = asg.getEngagedModules().iterator(); iterator.hasNext();) {
            AxisModule axisModule = (AxisModule) iterator.next();
            if (!getAxisConfig().isEngaged(axisModule)) {
                if (!SystemFilter.isFilteredOutModule(axisModule.getName())) {
                    engagedModules.add(new ModuleMetaData(axisModule.getName(),
                                                          axisModule.getVersion()));
                }
            }

        }
        return (ModuleMetaData[]) engagedModules.toArray(new ModuleMetaData[engagedModules.size()]);
    }

    /**
     * List engaged modules for a given operation
     *
     * @param serviceName
     * @param operationName
     * @return list of operation modules
     * @throws AxisFault
     */
    public ModuleMetaData[] listOperationModules(String serviceName,
                                                 String operationName) throws AxisFault {
        AxisService service = getAxisConfig().getService(serviceName);
        if (service == null) {
            throw new AxisFault("Service not found or invalid service");
        }
        AxisOperation operation = service.getOperation(new QName(operationName));
        if (operation == null) {
            throw new AxisFault("Operation name invalid");
        }
        ArrayList emodules = new ArrayList();
        for (Iterator engagedModules = operation.getEngagedModules().iterator();
             engagedModules.hasNext();) {
            AxisModule module = (AxisModule) engagedModules.next();

            if (!(service.isEngaged(module) ||
                  service.getParent().isEngaged(module) ||
                  getAxisConfig().isEngaged(module))) {
                if (!SystemFilter.isFilteredOutModule(module.getName())) {
                    emodules.add(new ModuleMetaData(module.getName(), module.getVersion()));
                }
            }
        }
        return (ModuleMetaData[]) emodules.toArray(new ModuleMetaData[emodules.size()]);
    }

    /**
     * Return all available module meta-data (not counts)
     *
     * @param moduleId
     * @param moduleVersion
     * @return info of the module
     * @throws AxisFault
     */
    public ModuleMetaData getModuleInfo(String moduleId,
                                        String moduleVersion) throws AxisFault {

        ModuleMetaData moduleMetaData = new ModuleMetaData();
        AxisModule axisModule = getAxisModule(moduleId, moduleVersion);
        moduleMetaData.setModulename(axisModule.getName());

        moduleMetaData.setModuleId(moduleId);//getModuleId(axisModule));

        moduleMetaData.setEngaged(getAxisConfig().isEngaged(axisModule) + "");

        String description = axisModule.getModuleDescription();

        if (description != null) {
            moduleMetaData.setDescription(description);
        } else {
            moduleMetaData.setDescription("No description found");
        }

        moduleMetaData.setModuleVersion(moduleVersion);

        return moduleMetaData;
    }

    /**
     * Return all accumulated data about this module
     *
     * @param moduleId
     * @param moduleVersion
     * @return status of the module
     * @throws AxisFault
     */
    public Object[] getModuleStatus(String moduleId,
                                    String moduleVersion) throws AxisFault {
        throw new AxisFault("Operation not implemented");
    }

    /**
     * Engaging a module globally
     *
     * @param moduleId
     * @param moduleVersion
     * @return The status
     * @throws AxisFault
     */
    public String engageModuleForSystem(String moduleId,
                                        String moduleVersion) throws AxisFault {

        // disengage from all services and operations
        AxisConfiguration axisConfig = getAxisConfig();
        String moduleName = Utils.getModuleName(moduleId, moduleVersion);
        AxisModule axisModule = axisConfig.getModule(moduleName);

        if (axisModule == null) {
            throw new AxisFault("Invalid module, " + moduleName);
        }

        if (axisConfig.isEngaged(axisModule)) {
            return Utils.getModuleName(moduleId, moduleVersion) + " module is already engaged globally.";
        }

        // Check for rahas engagement. Rahas requires Rampart also to be engaged
        if (moduleId.equalsIgnoreCase("rahas")) {
            AxisModule rampartModule = axisConfig.getModule(RAMPART);
            if (rampartModule == null) {
                throw new AxisFault("Rampart module not found! <br/> " +
                                    "In order to engage the Rahas module, " +
                                    "the Rampart module should be present.");
            }
            if (!axisConfig.isEngaged(rampartModule)) {
                engageModuleForSystem(rampartModule.getName(),
                                      rampartModule.getVersion());
            }
        }

        try {
            axisConfig.engageModule(axisModule);
        } catch (AxisFault axisFault) {
            log.error("Error occurred while " + Utils.getModuleName(moduleId, moduleVersion) +
                      " module was trying to be engaged to the service", axisFault);

            return "An error occurred while trying to engage the " +
                   Utils.getModuleName(moduleId, moduleVersion) + " module.\n" +
                   "This may be due to either the module being faulty, or the system, service(s) or" +
                   " operations(s) already being engaged to a different version of the " +
                   moduleId + " module. Please disengage this module from the system, service(s) or " +
                   "operation(s) and retry.";
        }

        // store the global engagement status
        ModuleDO moduleDO = pm.getModule(moduleId, moduleVersion);
        if (moduleDO != null) {
            moduleDO.setIsGloballyEngaged(true);
            try {
                pm.updateModule(moduleDO);
            } catch (ModuleNotFoundException e) {
                throw AxisFault.makeFault(e);
            }
        } else {
            log.warn("Module " + Utils.getModuleName(moduleId, moduleVersion) +
                     " not found in database");
        }

        // We should not engage the throttle module to the wso2wsas-administration service group
        disengageThrottleModuleFromAdminServices(moduleId, axisConfig, axisModule);

        return Utils.getModuleName(moduleId, moduleVersion) + " was successfully engaged globally.";
    }

    private void disengageThrottleModuleFromAdminServices(String moduleId,
                                                          AxisConfiguration axisConfig,
                                                          AxisModule axisModule) throws AxisFault {
        if (moduleId.equals("wso2throttle")) {
            AxisServiceGroup adminSG =
                    axisConfig.getServiceGroup(ServerConstants.ADMIN_SERVICE_GROUP);
            if (adminSG != null) {
                adminSG.disengageModule(axisModule);
            }
            AxisServiceGroup codegenSG =
                    axisConfig.getServiceGroup(ServerConstants.CODEGEN_SERVICE_GROUP);
            if (codegenSG != null) {
                codegenSG.disengageModule(axisModule);
            }
            AxisServiceGroup statSG =
                    axisConfig.getServiceGroup(ServerConstants.STATISTICS_SERVICE_GROUP);
            if (statSG != null) {
                statSG.disengageModule(axisModule);
            }
            AxisServiceGroup tracerSG =
                    axisConfig.getServiceGroup(ServerConstants.TRACER_SERVICE_GROUP);
            if (tracerSG != null) {
                tracerSG.disengageModule(axisModule);
            }
        }
    }

    /**
     * @param moduleId
     * @param moduleVersion
     * @return True if the module the engagement was successful, false otherwise.
     * @throws AxisFault
     */
    public boolean disengageModuleFromSystem(String moduleId,
                                             String moduleVersion) throws AxisFault {
        if (moduleId.equals("addressing")) {
            throw new AxisFault("addressing cannot be disengaged since this is a core module required " +
                                "by axis2 to support services deployed in the soapsession scope. " +
                                "Services deployed in this scope require the addressing module to " +
                                "be engaged. Also, the sandesha2 module requires addressing.");
        }
        AxisModule module = getAxisModule(moduleId, moduleVersion);
        try {
            AxisConfiguration axisConfig = getAxisConfig();
            if (axisConfig.isEngaged(module)) {
                if (moduleId.equalsIgnoreCase("rahas")) {
                    HashMap services = axisConfig.getServices();
                    for (Iterator iter = services.values().iterator(); iter.hasNext();) {
                        checkRahasDisengagingForService((AxisService) iter.next(), moduleId);
                    }
                } else if (moduleId.equalsIgnoreCase("rampart")) {

                    // Check whether it is possible to disengage Rampart
                    AxisModule rahasModule = axisConfig.getModule(RAHAS);
                    if (axisConfig.isEngaged(rahasModule)) {
                        throw new AxisFault("Cannot globally disengage Rampart since Rahas " +
                                            "is globally engaged. Disengage Rahas and try again.");
                    }

                    HashMap services = axisConfig.getServices();
                    for (Iterator iter = services.values().iterator(); iter.hasNext();) {
                        checkRampartDisengagingForService((AxisService) iter.next(), moduleId);
                    }
                }
                axisConfig.disengageModule(module);
            }
        } catch (Exception e) {
            log.error("Error occurred while disengaging module " + module.getName(), e);
            throw AxisFault.makeFault(e);
        }

        // store the global engagement status
        ModuleDO moduleDO = pm.getModule(moduleId, moduleVersion);
        if (moduleDO != null) {
            moduleDO.setIsGloballyEngaged(false);
            try {
                pm.updateModule(moduleDO);
            } catch (ModuleNotFoundException e) {
                log.error("Error occurred while trying to set the global disengagement status in " +
                          "the database of module " + module.getName(), e);
                throw AxisFault.makeFault(e);
            }
        } else {
            log.warn("Module " + moduleId + " not found in database while trying to globally " +
                     "disengage it");
        }

        return true;
    }

    private void checkRampartDisengagingForService(AxisService service,
                                                   String moduleId) throws AxisFault {
        // This check is not needed for admin service groups
        if (moduleId.equalsIgnoreCase("rampart")) {
            String serviceName = service.getName();

            // If there is a sec scenario, Rampart is needed
            ServiceDO serviceDO = pm.getService(serviceName,
                                                ServiceIdentifierDO.EMPTY_SERVICE_VERSION);
            if (serviceDO == null) {
                return;
            }
            if (serviceDO.getSecurityScenario() != null) {
                throw new AxisFault("Rampart cannot be disengaged since " +
                                    serviceName +
                                    " service is associated with a security scenario.");
            }

            // Check whether Rahas is engaged. If so, Rampart cannot be disengaged since Rahas
            // requires Rampart.
            AxisModule rahasModule = getAxisConfig().getModule(RAHAS);
            if (rahasModule != null) {
                if (service.isEngaged(rahasModule)) {
                    throw new AxisFault("Rampart cannot be disengaged since Rahas is engaged to " +
                                        serviceName + " service");
                }
                for (Iterator iter = service.getOperations(); iter.hasNext();) {
                    AxisOperation op = (AxisOperation) iter.next();
                    if (op.isEngaged(rahasModule)) {
                        throw new AxisFault("Rampart cannot be disengaged since Rahas is engaged to " +
                                            op.getName().getLocalPart() + " operation in " +
                                            serviceName + " service");
                    }
                }
            }
        }
    }

    private void checkRahasDisengagingForService(AxisService service,
                                                 String moduleId) throws AxisFault {
        if (moduleId.equalsIgnoreCase("rahas")) {
            String serviceName = service.getName();

            // If there is a sec scenario, Rahas may be needed
            ServiceDO serviceDO = pm.getService(serviceName,
                                                ServiceIdentifierDO.EMPTY_SERVICE_VERSION);
            if (serviceDO != null) {
                SecurityScenarioDO secScenario = serviceDO.getSecurityScenario();

                AxisModule rahasModule = getAxisConfig().getModule(RAHAS);
                ModuleDO moduleDO =
                        pm.getModule(rahasModule.getName(), rahasModule.getVersion());
                if (moduleDO != null && secScenario != null) {
                    if (secScenario.modules.contains(moduleDO)) {
                        throw new AxisFault("Rahas cannot be disengaged since " + serviceName +
                                            " service is associated with a security scenario which " +
                                            "requires the Rahas module.");

                    }
                }
            }
        }
    }

    /**
     * Delete a module
     *
     * @param moduleId
     * @param moduleVersion
     * @throws AxisFault
     */
    public void removeModule(String moduleId, String moduleVersion) throws AxisFault {

        // We cannot delete items from a URL repo
        String axis2Repo = ServerConfiguration.getInstance().
                getFirstProperty(ServerConfiguration.AXIS2_CONFIG_REPO_LOCATION);
        if (WsasUtils.isURL(axis2Repo)) {
            throw new AxisFault("You are not permitted to remove the " +
                                Utils.getModuleName(moduleId, moduleVersion) +
                                " module from the URL repository " + axis2Repo);
        }

        // We cannot remove modules which are required by security scenarios
        ModuleDO moduleDO = pm.getModule(moduleId, moduleVersion);
        if (moduleDO != null) {
            if (!moduleDO.getSecurityScenarios().isEmpty()) {
                throw new AxisFault("The " + Utils.getModuleName(moduleId, moduleVersion) +
                                    " module cannot be removed since it is " +
                                    "associated with at least one security scenario." +
                                    " These security scenarios can be applied to services. " +
                                    "Removing " + moduleId + " will make the system inconsistent.");
            }
        }

        // We cannot remove the addressing module if at least one service is deployed in soapsession scope
        if (moduleId.equals("addressing")) {
            throw new AxisFault("addressing cannot be removed since this is a core module required " +
                                "by axis2 to support services deployed in the soapsession scope. " +
                                "Services deployed in this scope require the addressing module to " +
                                "be engaged. Also, the sandesha2 module requires addressing.");
        }

        // Check whether this file can be deleted.
        // We should proceed only if this file can be deleted.
        String fileName = getAxisConfig().getModule(moduleId, moduleVersion).getFileName().getPath();
        File file = new File(fileName);
        if (!file.canWrite()) {
            throw new AxisFault("Module file deletion failed." +
                                " Due to a JVM issue on MS-Windows, module files " +
                                "cannot be deleted!");
        }

        disengageModuleFromSystem(moduleId, moduleVersion);

        // Delete the MAR file
        if (file.exists()) {
            if (!(file.isDirectory() &&
                  new FileManipulator().deleteDir(file))) {
                if (!file.delete()) {
                    throw new AxisFault("Module file deletion failed." +
                                        " Due to a JVM issue on MS-Windows, module files " +
                                        "cannot be deleted!");
                }
            }
        } else {
            throw new AxisFault("Module file " + file.getAbsolutePath() + " not found! ");
        }

        pm.deleteModule(moduleId, moduleVersion);
        getAxisConfig().removeModule(moduleId, moduleVersion);
    }

    /**
     * @param moduleId
     * @param moduleVersion
     * @param serviceGroupId
     * @return The status message
     * @throws AxisFault
     */
    public String engageModuleForServiceGroup(String moduleId,
                                              String moduleVersion,
                                              String serviceGroupId) throws AxisFault {

        AxisConfiguration axisConfig = getAxisConfig();
        AxisServiceGroup asg = axisConfig.getServiceGroup(serviceGroupId);
        if (asg == null) {
            throw new AxisFault("Service group not found");
        }
        AxisModule axisModule = getAxisModule(moduleId, moduleVersion);
        if (asg.isEngaged(axisModule)) {
            return "Module " + Utils.getModuleName(moduleId, moduleVersion) +
                   " has already been engaged to service group " + serviceGroupId;
        }

        // Check for rahas engagement. Rahas requires Rampart also to be engaged
        if (moduleId.equalsIgnoreCase("rahas")) {
            AxisModule rampartModule = axisConfig.getModule(RAMPART);
            if (rampartModule == null) {
                throw new AxisFault("Rampart module not found! <br/> " +
                                    "In order to engage the Rahas module, " +
                                    "the Rampart module should be present.");
            }
            if (!asg.isEngaged(rampartModule)) {
                engageModuleForServiceGroup(rampartModule.getName(),
                                            rampartModule.getVersion(),
                                            serviceGroupId);
            }
        }

        for (Iterator serviceIter = asg.getServices(); serviceIter.hasNext();) {
            AxisService service = (AxisService) serviceIter.next();
            if (service.isEngaged(axisModule)) {
                service.disengageModule(axisModule);

                // persist the service-module disengagement
                ServiceDO serviceDO = pm.getService(service.getName(),
                                                    ServiceIdentifierDO.EMPTY_SERVICE_VERSION);
                pm.removeEngagedServiceFromModule(moduleId, moduleVersion, serviceDO);
            }
        }
        asg.engageModule(axisModule);

        // Persist the service group-module engagement
        pm.addEngagedServiceGroupToModule(moduleId,
                                          moduleVersion,
                                          pm.getServiceGroup(serviceGroupId));
        return "Module " + Utils.getModuleName(moduleId, moduleVersion) +
               " successfully engaged to service group " + serviceGroupId;
    }

    /**
     * @param moduleId
     * @param moduleVersion
     * @param serviceGroupId
     * @throws AxisFault
     */
    public void disengageModuleFromServiceGroup(String moduleId,
                                                String moduleVersion,
                                                String serviceGroupId) throws AxisFault {
        AxisServiceGroup serviceGrp = getAxisConfig().getServiceGroup(serviceGroupId);
        if (serviceGrp == null) {
            throw new AxisFault("Service group " + serviceGroupId + " not found ");
        }
        AxisModule module = getAxisModule(moduleId, moduleVersion);
        if (serviceGrp.isEngaged(module)) {
            if (moduleId.equalsIgnoreCase("rahas")) { // Check if Rahas disengaging is possible
                for (Iterator iter = serviceGrp.getServices(); iter.hasNext();) {
                    checkRahasDisengagingForService((AxisService) iter.next(), moduleId);
                }
            } else
            if (moduleId.equalsIgnoreCase("rampart")) { // Check if rampart disengaging is possible
                for (Iterator iter = serviceGrp.getServices(); iter.hasNext();) {
                    checkRampartDisengagingForService((AxisService) iter.next(), moduleId);
                }
            }
            serviceGrp.disengageModule(module);

            // persist the serviceGrp-module disengagement
            pm.removeEngagedServiceGroupFromModule(moduleId, moduleVersion,
                                                   pm.getServiceGroup(serviceGroupId));
        }
    }

    /**
     * Engaging a module to a service
     *
     * @param moduleId
     * @param moduleVersion
     * @param serviceId
     * @param serviceVersion
     * @return A message
     * @throws AxisFault
     */
    public String engageModuleForService(String moduleId,
                                         String moduleVersion,
                                         String serviceId,
                                         String serviceVersion) throws AxisFault {

        AxisConfiguration axisConfig = getAxisConfig();
        AxisService axisService = axisConfig.getService(serviceId);
        if (axisService == null) {
            throw new AxisFault("Service not found");
        }
        AxisModule axisModule = getAxisModule(moduleId, moduleVersion);
        if (axisConfig.isEngaged(axisModule)) {
            return Utils.getModuleName(moduleId, moduleVersion) +
                   " module is globally engaged. You need not engage it to this service.";
        }

        if (axisService.isEngaged(axisModule)) {
            // Module is already engaged so just return the value;
            return Utils.getModuleName(moduleId, moduleVersion) +
                   " module is already engaged to the " + serviceId + " service";
        }

        // Check for rahas engagement. Rahas requires Rampart also to be engaged
        if (moduleId.equalsIgnoreCase("rahas")) {
            AxisModule rampartModule = axisConfig.getModule(RAMPART);
            if (rampartModule == null) {
                throw new AxisFault("Rampart module not found! <br/> " +
                                    "In order to engage the Rahas module, " +
                                    "the Rampart module should be present.");
            }
            if (!axisService.isEngaged(rampartModule)) {
                engageModuleForService(rampartModule.getName(),
                                       rampartModule.getVersion(),
                                       serviceId,
                                       serviceVersion);
            }
            setRahasParameters(serviceId, serviceVersion);
        }

        // Engage the module
        try {
            axisService.engageModule(axisModule);
        } catch (AxisFault e) {
            String msg = "Cannot engage module " + moduleId +
                         (moduleVersion != null ? "-" + moduleVersion : "") +
                         ". You may be trying to engage an older version of an already engaged " +
                         "module. Please disengage the newer version of the module and retry. ";
            log.warn(msg, e);
            return msg;
        }

        ServiceDO serviceDO = pm.getService(serviceId, serviceVersion);
        if (serviceDO != null) {
            for (Iterator iter = serviceDO.getEngagedModules().iterator();
                 iter.hasNext();) {
                ModuleDO oldModuleDO = (ModuleDO) iter.next();
                if (oldModuleDO.getModuleIdentifierDO().getName().equals(moduleId)) {
                    serviceDO.getEngagedModules().remove(oldModuleDO);
                    try {
                        pm.updateService(serviceDO);
                        pm.updateModule(oldModuleDO);
                    } catch (Exception e) {
                        throw AxisFault.makeFault(e);
                    }
                }
            }

            // persist the service-module engagement
            pm.addEngagedServiceToModule(moduleId, moduleVersion, serviceDO);

            return Utils.getModuleName(moduleId, moduleVersion) +
                   " module was successfully engaged to service " +
                   serviceId + (serviceVersion != null ? serviceVersion : "");
        }
        return serviceId + (serviceVersion != null ? serviceVersion : "") + " not found!";
    }

    private void setRahasParameters(String serviceId, String serviceVersion) throws AxisFault {
        ServiceDO serviceDO = pm.getService(serviceId, serviceVersion);
        if (serviceDO == null) {
            return;
        }
        KeyStoreDO pkStore = serviceDO.getPrivateKeyStore();
        Properties cryptoProps = new Properties();
        if (pkStore != null) {
            cryptoProps.setProperty(ServerCrypto.PROP_ID_KEY_STORE,
                                    pkStore.getFilePath());
            cryptoProps.setProperty(ServerCrypto.PROP_ID_DEFAULT_ALIAS,
                                    pkStore.getPrivateKeyAlias());
        }
        StringBuffer trustStores = new StringBuffer();
        for (Iterator tsIter = serviceDO.getTrustedCertStores().iterator();
             tsIter.hasNext();) {
            KeyStoreDO trustStore = (KeyStoreDO) tsIter.next();
            trustStores.append(trustStore.getFilePath()).append(",");
        }

        ServiceAdmin serviceAdmin = new ServiceAdmin();
        cryptoProps.setProperty(ServerCrypto.PROP_ID_TRUST_STORES, trustStores.toString());
        try {
            serviceAdmin.
                    setServiceParameter2(serviceId,
                                         RahasUtil.getSCTIssuerConfigParameter(ServerCrypto.class.getName(),
                                                                               cryptoProps,
                                                                               -1,
                                                                               null,
                                                                               true,
                                                                               true));
        } catch (ServerException e) {
            throw new AxisFault("Could not configure Rahas parameters", e);
        }
        serviceAdmin.setServiceParameter2(serviceId,
                                          RahasUtil.getTokenCancelerConfigParameter());
    }

    /**
     * @param moduleId
     * @param moduleVersion
     * @param serviceId
     * @param serviceVersion
     * @throws AxisFault
     */
    public void disengageModuleFromService(String moduleId,
                                           String moduleVersion,
                                           String serviceId,
                                           String serviceVersion) throws AxisFault {
        AxisService axisService = getAxisConfig().getService(serviceId);
        if (axisService == null) {
            throw new AxisFault("Service not found");
        }
        AxisModule module = getAxisModule(moduleId, moduleVersion);
        if (axisService.isEngaged(module)) {
            if (moduleId.equals("addressing") &&
                axisService.getScope().equals(org.apache.axis2.Constants.SCOPE_SOAP_SESSION)) {
                throw new AxisFault("addressing cannot be disengaged since the " + axisService.getName() +
                                    " service is deployed in the soapsession scope. Services deployed " +
                                    "in this scope require the addressing module to be engaged.");
            }

            // Check whether it is possible to disengage Rahas
            checkRahasDisengagingForService(axisService, moduleId);

            // Check whether it is possible to disengage Rampart
            checkRampartDisengagingForService(axisService, moduleId);

            axisService.disengageModule(module);

            // persist the service-module disengagement
            ServiceDO serviceDO = pm.getService(serviceId,
                                                ServiceIdentifierDO.EMPTY_SERVICE_VERSION);
            if (serviceDO != null) {
                pm.removeEngagedServiceFromModule(moduleId, moduleVersion,
                                                  serviceDO);
                removeRahasParameters(moduleId, serviceId);
            }
        }
    }

    /**
     * Forcefully disengage modules without taking into consideration any of the
     * restrictions.
     *
     * @param moduleId
     * @param moduleVersion
     * @param serviceId
     * @param serviceVersion
     * @throws AxisFault
     */
    public void forceDisengageModuleFromService(String moduleId,
                                                String moduleVersion,
                                                String serviceId,
                                                String serviceVersion) throws AxisFault {
        AxisService service = getAxisConfig().getService(serviceId);
        if (service == null) {
            throw new AxisFault("Service not found");
        }
        AxisModule module = getAxisModule(moduleId, moduleVersion);
        if (service.isEngaged(module)) {
            service.disengageModule(module);

            // persist the service-module disengagement
            ServiceDO serviceDO = pm.getService(serviceId,
                                                ServiceIdentifierDO.EMPTY_SERVICE_VERSION);
            if (serviceDO != null) {
                pm.removeEngagedServiceFromModule(moduleId, moduleVersion,
                                                  serviceDO);
                removeRahasParameters(moduleId, serviceId);
            }
        }
    }


    private void removeRahasParameters(String moduleId, String serviceId) throws AxisFault {
        if (moduleId.equals("rahas")) {

            //Remove the Rahas configuration parameters
            ServiceAdmin serviceAdmin = new ServiceAdmin();
            serviceAdmin.removeServiceParameter(serviceId,
                                                SCTIssuerConfig.SCT_ISSUER_CONFIG.
                                                        getLocalPart());
            serviceAdmin.removeServiceParameter(serviceId,
                                                TokenCancelerConfig.TOKEN_CANCELER_CONFIG.
                                                        getLocalPart());
        }
    }

    /**
     * Engaging a module to an operation
     *
     * @param moduleId
     * @param moduleVersion
     * @param serviceId
     * @param serviceVersion
     * @param operationId
     * @return A message
     * @throws AxisFault
     */
    public String engageModuleForOperation(String moduleId,
                                           String moduleVersion,
                                           String serviceId,
                                           String serviceVersion,
                                           String operationId) throws AxisFault {
        AxisConfiguration axisConfig = getAxisConfig();
        AxisService service = axisConfig.getService(serviceId);
        if (service == null) {
            throw new AxisFault("Service not found");
        }
        AxisOperation axisOp = service.getOperation(new QName(operationId));
        if (axisOp == null) {
            throw new AxisFault("Operation not found in the service : " + serviceId);
        }
        AxisModule axisModule = getAxisModule(moduleId, moduleVersion);
        if (axisConfig.isEngaged(axisModule)) {
            return Utils.getModuleName(moduleId, moduleVersion) +
                   " module is globally engaged. You need not engage it to this operation.";
        }

        // --------- if the module is already engaged to the service group, display a msg ------
        AxisServiceGroup axisServiceGroup = (AxisServiceGroup) axisOp.getParent().getParent();
        if (axisServiceGroup.isEngaged(axisModule)) {
            return Utils.getModuleName(moduleId, moduleVersion) +
                   " module is engaged to service group " + axisServiceGroup.getServiceGroupName() +
                   ". You need not engage it to this operation.";
        }

        // --------- if the module is already engaged to the service, display a msg ------
        AxisService axisService = (AxisService) axisOp.getParent();
        if (axisService.isEngaged(axisModule)) {
            return Utils.getModuleName(moduleId, moduleVersion) +
                   " module is engaged to service " + serviceId +
                   ". You need not engage it to this operation.";
        }

        // -------- if the modules is already engaged, display a message ---------
        if (axisOp.isEngaged(axisModule)) {
            return Utils.getModuleName(moduleId, moduleVersion) +
                   " module is already engaged to operation " + operationId;
        }

        // Check for rahas engagement. Rahas requires Rampart also to be engaged
        if (moduleId.equalsIgnoreCase("rahas")) {
            AxisModule rampartModule = axisConfig.getModule(RAMPART);
            if (rampartModule == null) {
                throw new AxisFault("Rampart module not found! <br/> " +
                                    "In order to engage the Rahas module, " +
                                    "the Rampart module should be present.");
            }
            if (!axisOp.isEngaged(rampartModule)) {
                engageModuleForOperation(rampartModule.getName(),
                                         rampartModule.getVersion(),
                                         serviceId,
                                         serviceVersion,
                                         operationId);
            }
        }

        // ------------------- Do Engagement -------------------------------------
        try {
            axisOp.engageModule(getAxisConfig().getModule(axisModule.getName(),
                                                          axisModule.getVersion()));
        } catch (AxisFault axisFault) {
            //For the moment we assume that this is because we tried to engage an
            // already engaged module so we ignore it
            String msg = "Error while trying to engage module  " + moduleId +
                         "-" + moduleVersion + " to axis operation " + serviceId + "#" +
                         operationId;
            log.warn(msg, axisFault);
            return msg;
        }

        // persist the service-module engagement
        pm.addEngagedOperationToModule(moduleId, moduleVersion,
                                       pm.getOperation(serviceId,
                                                       ServiceIdentifierDO.EMPTY_SERVICE_VERSION,
                                                       axisOp.getName().getLocalPart()));

        return Utils.getModuleName(moduleId, moduleVersion) +
               " module has been successfully engaged to operation " + operationId;
    }

    /**
     * @param moduleId
     * @param moduleVersion
     * @param serviceId
     * @param serviceVersion
     * @param operationId
     * @throws AxisFault
     */
    public void disengageModuleFromOperation(String moduleId,
                                             String moduleVersion,
                                             String serviceId,
                                             String serviceVersion,
                                             String operationId) throws AxisFault {
        AxisService service = getAxisConfig().getService(serviceId);

        if (service == null) {
            throw new AxisFault("Service not found");
        }
        AxisOperation axisOp = service.getOperation(new QName(operationId));
        if (axisOp == null) {
            throw new AxisFault("Operation not found in the service : " + serviceId);
        }

        // Rampart cannot be disengaged, if Rahas is engaged
        if (moduleId.equalsIgnoreCase("rampart")) {
            AxisModule rahasModule = getAxisConfig().getModule(RAHAS);
            if (rahasModule != null) {
                if (axisOp.isEngaged(rahasModule)) {
                    throw new AxisFault("Rampart cannot be disengaged since Rahas " +
                                        "is engaged to operation " + operationId +
                                        ". Disengage Rahas and try again.");
                }
            }
        }
        AxisModule module = getAxisModule(moduleId, moduleVersion);
        if (axisOp.isEngaged(module)) {
            checkRampartDisengagingForService((AxisService) axisOp.getParent(),
                                              moduleId);
            axisOp.disengageModule(module);

            // persist the service-module disengagement
            pm.removeEngagedOperationFromModule(moduleId, moduleVersion,
                                                pm.getOperation(serviceId,
                                                                ServiceIdentifierDO.EMPTY_SERVICE_VERSION,
                                                                axisOp.getName().getLocalPart()));
        }
    }

    /**
     * Engaging a module to a messageE
     *
     * @param moduleId
     * @param moduleVersion
     * @param serviceId
     * @param serviceVersion
     * @param operationId
     * @param messageLabel
     * @throws AxisFault
     */
    public void engageModuleForMessage(String moduleId,
                                       String moduleVersion,
                                       String serviceId,
                                       String serviceVersion,
                                       String operationId,
                                       String messageLabel) throws AxisFault {
        throw new AxisFault("Not implemented");
    }

    /**
     * Return all parameters for this module (including inherited ones),
     * where each parameter is an XML fragment representing the "parameter" element
     *
     * @param moduleId
     * @param moduleVersion
     * @return OMElement
     */
    public OMElement[] getModuleParameters(String moduleId,
                                           String moduleVersion) throws AxisFault {
        AxisModule module = getAxisModule(moduleId, moduleVersion);
        ArrayList parameters = module.getParameters();
        OMElement[] paraElements = new OMElement[parameters.size()];

        for (int i = 0; i < parameters.size(); i++) {
            Parameter parameter = (Parameter) parameters.get(i);
            paraElements[i] = parameter.getParameterElement();
        }

        return paraElements;
    }

    /**
     * @param moduleId
     * @param moduleVersion
     * @return OMElement
     * @throws AxisFault
     */
    public OMElement getPolicy(String moduleId,
                               String moduleVersion) throws AxisFault {
        AxisModule module = getAxisModule(moduleId, moduleVersion);

        PolicyInclude modulePolicyInclude = module.getPolicyInclude();
        Policy modulePolicy = modulePolicyInclude.getPolicy();

        if (modulePolicy == null) {
            return PolicyUtil.getEmptyPolicyAsOMElement();
        }

        return PolicyUtil.getPolicyAsOMElement(modulePolicy);
    }

    /**
     * @param moduleId
     * @param moduleVersion
     * @param policyString
     * @throws AxisFault
     */
    public void setPolicy(String moduleId,
                          String moduleVersion,
                          String policyString) throws AxisFault {

        ByteArrayInputStream bais = new ByteArrayInputStream(policyString.getBytes());
        OMElement policyElement;
        try {
            policyElement = new StAXOMBuilder(bais).getDocumentElement();
            policyElement.build();
        } catch (Exception e) {
            throw new AxisFault("Cannot deserialize module policy. The policy may be invalid.", e);
        }

        AxisModule axisModule = getAxisModule(moduleId, moduleVersion);
        Policy modulePolicy;

        // Update the module policy
        try {
            modulePolicy = PolicyUtil.getPolicyFromOMElement(policyElement);

            // DB update
            ModulePolicyDO modulePolicyDO = new ModulePolicyDO();
            ModuleDO moduleDO = new ModuleDO();
            ModuleIdentifierDO mid = new ModuleIdentifierDO();
            mid.setName(moduleId);
            mid.setVersion(moduleVersion);
            moduleDO.setModuleIdentifierDO(mid);
            modulePolicyDO.setModule(moduleDO);

            OMAttribute attribute =
                    policyElement.getAttribute(new QName(Constants.URI_WSU_NS, Constants.ATTR_ID));
            if (attribute == null) {
                throw new AxisFault("ID Attribute not found in Policy element. " +
                                    "Please verify that the policy XML is valid.");
            }
            modulePolicyDO.setUuid(attribute.getAttributeValue());

            modulePolicyDO.setPolicy(policyElement.toString());
            pm.updateModulePolicy(modulePolicyDO);

        } catch (ModulePolicyNotFoundException e) {
            throw new AxisFault("Module Policy not found in DB", e);
        }

        axisModule.getPolicyInclude().updatePolicy(modulePolicy);
    }

    /**
     * Return only the parameters for explicitly set for this module (not including inherited ones),
     * where each parameter is an XML fragment representing the "parameter" element
     *
     * @param moduleId
     * @param moduleVersion
     * @return The declared parameter
     * @throws AxisFault
     */
    public OMElement[] getDeclaredModuleParameters(String moduleId,
                                                   String moduleVersion) throws AxisFault {
        AxisModule module = getAxisModule(moduleId, moduleVersion);
        ArrayList parameters = module.getParameters();
        ArrayList returnValues = new ArrayList();
        for (int i = 0; i < parameters.size(); i++) {
            Parameter parameter = (Parameter) parameters.get(i);
            OMElement element = parameter.getParameterElement();
            if (element != null) {
                returnValues.add(element);
            }
        }
        return (OMElement[]) returnValues.toArray(new OMElement[returnValues.size()]);
    }

    /**
     * @param moduleId
     * @param moduleVersion
     * @param paramterElements
     * @throws AxisFault
     */
    public void setModuleParameters(String moduleId, String moduleVersion,
                                    OMElement[] paramterElements) throws AxisFault {
        for (int i = 0; i < paramterElements.length; i++) {
            setModuleParameter(moduleId, moduleVersion, paramterElements[i]);
        }

    }

    /**
     * @param moduleId
     * @param moduleVersion
     * @param paramterElement
     * @throws AxisFault
     */
    public void setModuleParameter(String moduleId, String moduleVersion,
                                   OMElement paramterElement) throws AxisFault {
        AxisModule axisModule = getAxisModule(moduleId, moduleVersion);

        // Persist the parameter
        ModuleDO moduleDO = pm.getModule(moduleId, moduleVersion);

        Parameter parameter = ParameterUtil.createParameter(paramterElement);
        Parameter p = axisModule.getParameter(parameter.getName());

        if (p != null) {
            if (!p.isLocked()) {
                axisModule.addParameter(parameter);

                ModuleParameterDO paramDO = pm.getModuleParameter(moduleDO,
                                                                  parameter.getName());
                paramDO.setValue(paramterElement.toString());
                pm.updateEntity(paramDO);
            }
        } else {
            axisModule.addParameter(parameter);

            ModuleParameterDO param = new ModuleParameterDO();
            param.setName(parameter.getName());
            param.setValue(paramterElement.toString());
            param.setModule(moduleDO);

            try {
                pm.addEntity(param);
            } catch (DuplicateEntityException e) {
                log.error("Module Parameter already exists", e);
            }
        }
    }

    /**
     * @param moduleId
     * @param moduleVersion
     * @param paramterElement
     * @throws AxisFault
     */
    public void removeModuleParameter(String moduleId, String moduleVersion,
                                      OMElement paramterElement) throws AxisFault {
        AxisModule axisModule = getAxisModule(moduleId, moduleVersion);

        // un-Persist the parameter
        ModuleDO moduleDO = pm.getModule(moduleId, moduleVersion);

        Parameter parameter = ParameterUtil.createParameter(paramterElement);
        Parameter p = axisModule.getParameter(parameter.getName());

        if (p != null) {
            if (!p.isLocked()) {
                axisModule.removeParameter(parameter);

                ModuleParameterDO paramDO = pm.getModuleParameter(moduleDO,
                                                                  parameter.getName());
                pm.deleteEntity(paramDO);
            }
        } else {
            axisModule.removeParameter(parameter);
        }
    }

    /**
     * To set module has default version for that module family
     *
     * @param moduleName
     * @param version
     * @throws AxisFault
     */
    public void setAs(String moduleName, String version) throws AxisFault {
        getAxisConfig().addDefaultModuleVersion(moduleName, version);
    }

    /**
     * @param moduleId
     * @param moduleVersion
     * @return PolicyData
     * @throws AxisFault
     */
    public PolicyData[] getPolicies(String moduleId, String moduleVersion) throws AxisFault {
        AxisModule axisModule =
                getAxisConfig().getModule(moduleId, moduleVersion);

        if (axisModule == null) {
            throw new AxisFault("Invalid module name");
        }

        ArrayList policyDataArray = new ArrayList();

        PolicyInclude policyInclude = axisModule.getPolicyInclude();
        PolicyRegistry registry = policyInclude.getPolicyRegistry();

        // module.xml
        List policyList = policyInclude.getPolicyElements(PolicyInclude.AXIS_MODULE_POLICY);

        if (!policyList.isEmpty()) {
            PolicyData policyData = new PolicyData();
            policyData.setWrapper("Policies in module.xml");
            policyData.setPolycies(PolicyUtil.processPolicyElements(policyList.iterator(), registry));
            policyDataArray.add(policyData);
        }

        return (PolicyData[]) policyDataArray.toArray(new PolicyData[policyDataArray.size()]);
    }

    private AxisModule getAxisModule(String moduleId,
                                     String moduleVersion) throws AxisFault {
        AxisModule module = getAxisConfig().getModule(moduleId, moduleVersion);

        if (module == null) {
            throw new AxisFault("Invalid module name : " +
                                Utils.getModuleName(moduleId, moduleVersion));
        }

        return module;
    }

    /**
     * List engaged modules of an operation's parent (i.e. AxisService)
     *
     * @param serviceName
     * @return Module data
     * @throws AxisFault
     */
    public ModuleMetaData[] listServicesParentEngagedModules(String serviceName) throws AxisFault {
        AxisService service = getAxisConfig().getService(serviceName);

        if (service == null) {
            throw new AxisFault("Service not found or invalid service");
        }

        List emodules = new ArrayList();
        for (Iterator engagedModules = service.getEngagedModules().iterator();
             engagedModules.hasNext();) {
            AxisModule module = (AxisModule) engagedModules.next();

            if (!getAxisConfig().isEngaged(module) &&
                service.getParent().isEngaged(module)) {
                if (!SystemFilter.isFilteredOutModule(module.getName())) {
                    emodules.add(new ModuleMetaData(module.getName(), module.getVersion()));
                }
            }
        }

        return (ModuleMetaData[]) emodules.toArray(new ModuleMetaData[emodules.size()]);

    }

    /**
     * List engaged modules for a given operation's parent's parent (i.e. the AxisServiceGroup)
     *
     * @param serviceName
     * @param operationName
     * @return Module data
     * @throws AxisFault
     */
    public ModuleMetaData[] listOperationsParentParentEngagedModules(String serviceName,
                                                                     String operationName)
            throws AxisFault {
        AxisService service = getAxisConfig().getService(serviceName);
        if (service == null) {
            throw new AxisFault("Service not found or invalid service");
        }

        AxisOperation operation = service.getOperation(new QName(operationName));

        if (operation == null) {
            throw new AxisFault("Operation name invalid");
        }

        ArrayList emodules = new ArrayList();

        for (Iterator engagedModules = operation.getEngagedModules().iterator();
             engagedModules.hasNext();) {
            AxisModule module = (AxisModule) engagedModules.next();

            if (service.getParent().isEngaged(module) &&
                !getAxisConfig().isEngaged(module)) {
                if (!SystemFilter.isFilteredOutModule(module.getName())) {
                    emodules.add(new ModuleMetaData(module.getName(), module.getVersion()));
                }
            }
        }

        return (ModuleMetaData[]) emodules.toArray(new ModuleMetaData[emodules.size()]);
    }

}
