/*
 * Copyright 2005-2007 WSO2, Inc. (http://wso2.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wso2.wsas.admin.service;

import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.OMNode;
import org.apache.axiom.om.OMText;
import org.apache.axis2.AxisFault;
import org.apache.axis2.context.ConfigurationContext;
import org.apache.axis2.context.MessageContext;
import org.apache.axis2.context.ServiceGroupContext;
import org.apache.axis2.deployment.DeploymentEngine;
import org.apache.axis2.description.AxisModule;
import org.apache.axis2.description.Parameter;
import org.apache.axis2.description.PolicyInclude;
import org.apache.axis2.engine.Handler;
import org.apache.axis2.engine.Phase;
import org.apache.axis2.phaseresolver.PhaseMetadata;
import org.apache.axis2.util.JavaUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.neethi.Policy;
import org.wso2.utils.AbstractAdmin;
import org.wso2.wsas.ServerConstants;
import org.wso2.wsas.admin.service.util.AdminAuthenticator;
import org.wso2.wsas.persistence.PersistenceManager;
import org.wso2.wsas.persistence.dataobject.ServiceUserDO;
import org.wso2.wsas.persistence.exception.ServiceUserNotFoundException;
import org.wso2.wsas.util.ParameterUtil;
import org.wso2.wsas.util.PolicyUtil;

import javax.activation.DataHandler;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;


/**
 * Admin service to manage operations on the <code>AxisConfiguration</code>
 */
public class GlobalAdmin extends AbstractAdmin {

    private static final Log log = LogFactory.getLog(GlobalAdmin.class);
    private PersistenceManager pm = new PersistenceManager();

    /**
     * Restore the default system settings and override all the changes persisted in the
     * database. Please note that this is possible only for the default database included
     * with WSO2 WSAS
     */
    public void restoreDefaults() {
        pm.restoreDefaults();
    }

    /**
     * List handlers in a given phase (global)
     *
     * @param flow
     * @param phaseName
     * @return list of global phase handlers as a String[].
     * @throws AxisFault
     */
    public String[] listGlobalPhaseHandlers(int flow, String phaseName) throws AxisFault {
        String[] handlers = null;
        switch (flow) {
            case PhaseMetadata.IN_FLOW: {
                ArrayList inflow = getAxisConfig().getInFlowPhases();
                for (int i = 0; i < inflow.size(); i++) {
                    Phase phase = (Phase) inflow.get(i);
                    if (phase.getPhaseName().equals(phaseName)) {
                        handlers = new String[phase.getHandlerCount()];
                        ArrayList hands = phase.getHandlers();
                        for (int j = 0; j < hands.size(); j++) {
                            Handler handler = (Handler) hands.get(j);
                            handlers[j] = handler.getName();
                        }
                    }
                }
                break;
            }
            case PhaseMetadata.OUT_FLOW: {
                ArrayList phases = getAxisConfig().getOutFlowPhases();
                for (int i = 0; i < phases.size(); i++) {
                    Phase phase = (Phase) phases.get(i);
                    if (phase.getPhaseName().equals(phaseName)) {
                        handlers = new String[phase.getHandlerCount()];
                        ArrayList hands = phase.getHandlers();
                        for (int j = 0; j < hands.size(); j++) {
                            Handler handler = (Handler) hands.get(j);
                            handlers[j] = handler.getName();
                        }
                    }
                }
                break;
            }
            case PhaseMetadata.FAULT_IN_FLOW: {
                ArrayList inflow = getAxisConfig().getInFaultFlowPhases();
                for (int i = 0; i < inflow.size(); i++) {
                    Phase phase = (Phase) inflow.get(i);
                    if (phase.getPhaseName().equals(phaseName)) {
                        handlers = new String[phase.getHandlerCount()];
                        ArrayList hands = phase.getHandlers();
                        for (int j = 0; j < hands.size(); j++) {
                            Handler handler = (Handler) hands.get(j);
                            handlers[j] = handler.getName();
                        }
                    }
                }
                break;
            }
            case PhaseMetadata.FAULT_OUT_FLOW: {
                ArrayList inflow = getAxisConfig().getOutFaultFlowPhases();
                for (int i = 0; i < inflow.size(); i++) {
                    Phase phase = (Phase) inflow.get(i);
                    if (phase.getPhaseName().equals(phaseName)) {
                        handlers = new String[phase.getHandlerCount()];
                        ArrayList hands = phase.getHandlers();
                        for (int j = 0; j < hands.size(); j++) {
                            Handler handler = (Handler) hands.get(j);
                            handlers[j] = handler.getName();
                        }
                    }
                }
                break;
            }
        }
        return handlers;
    }

    /**
     * return only the parameters for explicitly set for this module (not including inherited ones),
     * where each parameter is an XML fragment representing the "parameter" element.
     * Note: In this case (unlike module/service group etc.), there are no inherited parameters.
     */
    public OMElement[] getDeclaredSystemParameters() throws AxisFault {
        ArrayList parameters = getAxisConfig().getParameters();
        OMElement[] paraElements = new OMElement[parameters.size()];

        for (int i = 0; i < parameters.size(); i++) {
            Parameter parameter = (Parameter) parameters.get(i);
            paraElements[i] = parameter.getParameterElement();
        }

        return paraElements;
    }

    public void setSystemParameter(OMElement parameterElement) throws AxisFault {
        Parameter parameter = ParameterUtil.createParameter(parameterElement);
        getAxisConfig().addParameter(parameter);
    }

    /**
     * List global phases
     *
     * @param flow
     * @return Global phases
     * @throws AxisFault
     */
    public String[] listGlobalPhases(int flow) throws AxisFault {
        String[] phaseNames = null;

        switch (flow) {
            case PhaseMetadata.IN_FLOW: {
                ArrayList inflow = getAxisConfig().getInFlowPhases();
                phaseNames = new String[inflow.size()];

                for (int i = 0; i < inflow.size(); i++) {
                    Phase phase = (Phase) inflow.get(i);
                    phaseNames[i] = phase.getPhaseName();
                }

                break;
            }
            case PhaseMetadata.OUT_FLOW: {
                ArrayList inflow = getAxisConfig().getOutFlowPhases();
                phaseNames = new String[inflow.size()];

                for (int i = 0; i < inflow.size(); i++) {
                    Phase phase = (Phase) inflow.get(i);
                    phaseNames[i] = phase.getPhaseName();
                }

                break;
            }
            case PhaseMetadata.FAULT_IN_FLOW: {
                ArrayList inflow = getAxisConfig().getInFaultFlowPhases();
                phaseNames = new String[inflow.size()];

                for (int i = 0; i < inflow.size(); i++) {
                    Phase phase = (Phase) inflow.get(i);
                    phaseNames[i] = phase.getPhaseName();
                }

                break;
            }
            case PhaseMetadata.FAULT_OUT_FLOW: {
                ArrayList inflow = getAxisConfig().getOutFaultFlowPhases();
                phaseNames = new String[inflow.size()];

                for (int i = 0; i < inflow.size(); i++) {
                    Phase phase = (Phase) inflow.get(i);
                    phaseNames[i] = phase.getPhaseName();
                }

                break;
            }
        }
        return phaseNames;
    }

    public boolean login(String username, String password) throws AxisFault {

        MessageContext msgCtx = MessageContext.getCurrentMessageContext();

        ServiceGroupContext sgCtx = msgCtx.getServiceGroupContext();
        if ((username == null) || (password == null) ||
            username.trim().equals("") || password.trim().equals("")) {
            sgCtx.setProperty(ServerConstants.ADMIN_LOGGED_IN, "false");
            return false;
        }
        boolean loggedIn =
                new AdminAuthenticator().authenticate(username, password);
        SimpleDateFormat date = new SimpleDateFormat("'['yyyy-MM-dd HH:mm:ss,SSSS']'");
        Date currentTime = Calendar.getInstance().getTime();
        ServiceUserDO user = pm.getUser(username);
        String remoteIP = (String) msgCtx.getProperty(MessageContext.REMOTE_ADDR);
        if (loggedIn) {
            sgCtx.setProperty(ServerConstants.ADMIN_LOGGED_IN, "true");
            log.info("Administrator \'" + username + "' logged in at " + date.format(currentTime) +
                     " from IP address " + remoteIP);
            if (user.getLastLoggedIn() != null) {
                log.info("Last successful login at " + date.format(user.getLastLoggedIn()) +
                         " from IP Address " + user.getLastLoginIP());
            }
            if (user.getLastFailedLogIn() != null) {
                log.info("Last failed login at " + date.format(user.getLastFailedLogIn()) +
                         " from IP Address " + user.getLastFailedLoginIP());
            }
            user.setLastLoggedIn(currentTime);
            user.setLastLoginIP(remoteIP);
            try {
                pm.updateUser(user);
            } catch (ServiceUserNotFoundException e) {
                throw AxisFault.makeFault(e);
            }
            return true;
        } else {
            sgCtx.removeProperty(ServerConstants.ADMIN_LOGGED_IN);
            if (user != null) {
                log.warn("Failed login attempt by Administrator \'" + username + "\' at " +
                         date.format(currentTime) + " from IP address " + remoteIP);
                user.setLastFailedLogIn(currentTime);
                user.setLastFailedLoginIP(remoteIP);
                try {
                    pm.updateUser(user);
                } catch (ServiceUserNotFoundException e) {
                    throw AxisFault.makeFault(e);
                }
            } else {
                log.warn("Failed Administrator login attempt by non-existent Administrator \'" +
                         username + "\' at " + date.format(currentTime) +
                         " from IP address " + remoteIP);
            }
            return false;
        }
    }

    public void serverRegistered() {
        ConfigurationContext configCtx =
                MessageContext.getCurrentMessageContext().getConfigurationContext();
        configCtx.setProperty(ServerConstants.SERVER_REGISTERED, "true");
        pm.updateConfigurationProperty(ServerConstants.SERVER_REGISTERED, "true");
    }

    public boolean isServerRegistered() {
        ConfigurationContext configCtx =
                MessageContext.getCurrentMessageContext().getConfigurationContext();
        String serverReg = (String) configCtx.getProperty(ServerConstants.SERVER_REGISTERED);

        if (serverReg == null) {
            serverReg = pm.getConfigurationProperty(ServerConstants.SERVER_REGISTERED);
            if (serverReg == null) {
                serverReg = "false";
                pm.updateConfigurationProperty(ServerConstants.SERVER_REGISTERED,
                                               "false");
                configCtx.setProperty(ServerConstants.SERVER_REGISTERED, "false");
            }
        }

        return serverReg.equals("true");
    }

    public void logout() throws AxisFault {
        MessageContext.getCurrentMessageContext().
                getServiceGroupContext().getProperties().clear();
    }

    public boolean deployService(OMElement element) throws AxisFault {
        try {
            if (element == null) {
                throw new AxisFault("Element is null");
            }

            String fileName = element.getLocalName();
            OMNode node = element.getFirstOMChild();
            if (node instanceof OMText) {
                OMText txt = (OMText) node;
                String repo = getAxisConfig().getRepository().getPath();
                DataHandler expectedDH = (DataHandler) txt.getDataHandler();
                File serviceDir = new File(repo, "services");
                File file = new File(serviceDir, fileName);
                OutputStream out = new FileOutputStream(file);
                expectedDH.writeTo(out);
                out.flush();
                out.close();

                return true;
            } else {
                throw new AxisFault("Invalid Message");
            }
        } catch (IOException e) {
            throw AxisFault.makeFault(e);
        }
    }

    public boolean deployModule(OMElement element) throws AxisFault {
        try {
            if (element == null) {
                throw new AxisFault("Element null");
            }

            String fileName = element.getLocalName();
            OMNode node = element.getFirstOMChild();

            if (node instanceof OMText) {
                OMText txt = (OMText) node;
                String repo = getAxisConfig().getRepository().getPath();
                DataHandler expectedDH = (DataHandler) txt.getDataHandler();
                File moduleDir = new File(repo, "modules");
                File file = new File(moduleDir, fileName);
                OutputStream out = new FileOutputStream(file);
                expectedDH.writeTo(out);
                out.flush();
                out.close();

                AxisModule module =
                        DeploymentEngine.buildModule(file, getAxisConfig());
                getAxisConfig().addModule(module);

                return true;
            } else {
                throw new AxisFault("Invalid Message");
            }
        } catch (IOException e) {
            throw AxisFault.makeFault(e);
        }
    }

    public OMElement getPolicy() throws AxisFault {
        PolicyInclude globalPolicyInclude = getAxisConfig().getPolicyInclude();
        Policy globalPolicy = globalPolicyInclude.getPolicy();

        if (globalPolicy == null) {
            return PolicyUtil.getEmptyPolicyAsOMElement();
        }

        return PolicyUtil.getPolicyAsOMElement(globalPolicy);
    }

    public void setPolicy(OMElement policyElement) throws AxisFault {
        Policy globalPolicy;

        try {
            globalPolicy = PolicyUtil.getPolicyFromOMElement(policyElement);
        } catch (Exception ex) {
            throw new AxisFault("Cannot convert the OMElement to Policy", ex);
        }

        PolicyInclude globalPolicyInclude = getAxisConfig().getPolicyInclude();
        globalPolicyInclude.setPolicy(globalPolicy);
    }

    /**
     * This has used to test if the server is availabele and the current admins login status.
     *
     * @return boolean
     */
    public boolean ping() {
        MessageContext msgCtx = MessageContext.getCurrentMessageContext();
        ServiceGroupContext serviceGroupContext = msgCtx.getServiceGroupContext();
        Object property = serviceGroupContext.getProperty(ServerConstants.ADMIN_LOGGED_IN);
        if (property == null) {
            return false;
        } else {
            String status = (String) property;
            return JavaUtils.isTrue(status);
        }

    }
}
